@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            border-left: 4px solid #667eea;
        }
        .form-section h5 {
            color: #495057;
            border-bottom: 2px solid #dee2e6;
            padding-bottom: 10px;
            margin-bottom: 20px;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .logo-upload-container {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        .logo-upload-container:hover {
            border-color: #667eea;
            background: #f8f9fa;
        }
        .logo-preview {
            max-width: 200px;
            max-height: 200px;
            border-radius: 10px;
            display: none;
        }
        .switch-label {
            cursor: pointer;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Brands</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('brands.index') }}">Brands</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Add New Brand</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('brands.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to List
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Add New Brand</h4>
                        
                        @if($errors->any())
                            <div class="alert alert-danger">
                                <ul class="mb-0">
                                    @foreach($errors->all() as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                </ul>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('brands.store') }}" enctype="multipart/form-data" id="brandForm">
                            @csrf
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h5><i class="bx bx-info-circle"></i> Basic Information</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Brand Name</label>
                                        <input type="text" name="name" class="form-control" 
                                               value="{{ old('name') }}" required 
                                               placeholder="Enter brand name">
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Slug</label>
                                        <input type="text" name="slug" class="form-control" 
                                               value="{{ old('slug') }}" 
                                               placeholder="brand-name (auto-generated)">
                                        <div class="form-text">URL-friendly version of the name</div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Description</label>
                                    <textarea name="description" class="form-control" rows="3" 
                                              placeholder="Brand description">{{ old('description') }}</textarea>
                                </div>
                            </div>

                            <!-- Brand Logo -->
                            <div class="form-section">
                                <h5><i class="bx bx-image"></i> Brand Logo</h5>
                                
                                <div class="logo-upload-container mb-3" onclick="$('#logoInput').click()">
                                    <input type="file" name="logo" id="logoInput" class="d-none" accept="image/*">
                                    <i class="bx bx-cloud-upload fs-1 text-muted"></i>
                                    <p class="mb-0">Click to upload logo</p>
                                    <small class="text-muted">Recommended: 300x300px, PNG/JPG format</small>
                                </div>
                                
                                <img id="logoPreview" class="logo-preview" alt="Logo Preview">
                            </div>

                            <!-- Contact Information -->
                            <div class="form-section">
                                <h5><i class="bx bx-contact"></i> Contact Information</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Website</label>
                                        <input type="url" name="website" class="form-control" 
                                               value="{{ old('website') }}" 
                                               placeholder="https://example.com">
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Contact Email</label>
                                        <input type="email" name="contact_email" class="form-control" 
                                               value="{{ old('contact_email') }}" 
                                               placeholder="contact@brand.com">
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Contact Phone</label>
                                        <input type="text" name="contact_phone" class="form-control" 
                                               value="{{ old('contact_phone') }}" 
                                               placeholder="+1 (555) 123-4567">
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Country</label>
                                        <select name="country" class="form-select">
                                            <option value="">Select Country</option>
                                            @foreach($countries as $code => $name)
                                                <option value="{{ $name }}" {{ old('country') == $name ? 'selected' : '' }}>
                                                    {{ $name }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Established Year</label>
                                        <input type="number" name="established_year" class="form-control" 
                                               value="{{ old('established_year') }}" 
                                               min="1800" max="{{ date('Y') }}" 
                                               placeholder="2020">
                                    </div>
                                </div>
                            </div>

                            <!-- Categories -->
                            <div class="form-section">
                                <h5><i class="bx bx-category"></i> Associated Categories</h5>
                                
                                <div class="mb-3">
                                    <label class="form-label">Select Categories</label>
                                    <select name="category_ids[]" class="form-select" multiple size="5">
                                        @foreach($categories as $category)
                                            <option value="{{ $category->id }}" 
                                                    {{ in_array($category->id, old('category_ids', [])) ? 'selected' : '' }}>
                                                {{ $category->name }}
                                                @if($category->parent)
                                                    ({{ $category->parent->name }})
                                                @endif
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="form-text">Hold Ctrl/Cmd to select multiple categories</div>
                                </div>
                            </div>

                            <!-- Status & Options -->
                            <div class="form-section">
                                <h5><i class="bx bx-cog"></i> Status & Options</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Status</label>
                                        <select name="status" class="form-select" required>
                                            <option value="active" {{ old('status', 'active') == 'active' ? 'selected' : '' }}>
                                                Active
                                            </option>
                                            <option value="inactive" {{ old('status') == 'inactive' ? 'selected' : '' }}>
                                                Inactive
                                            </option>
                                        </select>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <div class="form-check form-switch mt-4">
                                            <input class="form-check-input" type="checkbox" name="is_featured" 
                                                   id="isFeatured" value="1" 
                                                   {{ old('is_featured') ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="isFeatured">
                                                Featured Brand
                                            </label>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Display Order</label>
                                        <input type="number" name="order" class="form-control" 
                                               value="{{ old('order', 0) }}" 
                                               min="0" placeholder="0">
                                        <div class="form-text">Lower numbers appear first</div>
                                    </div>
                                </div>
                            </div>

                            <!-- SEO Settings -->
                            <div class="form-section">
                                <h5><i class="bx bx-search"></i> SEO Settings</h5>
                                
                                <div class="mb-3">
                                    <label class="form-label">Meta Title</label>
                                    <input type="text" name="meta_title" class="form-control" 
                                           value="{{ old('meta_title') }}" 
                                           placeholder="Brand Name - Product Category">
                                    <div class="form-text">Title for search engines (50-60 characters)</div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Meta Description</label>
                                    <textarea name="meta_description" class="form-control" rows="2" 
                                              placeholder="Description for search engines">{{ old('meta_description') }}</textarea>
                                    <div class="form-text">Description for search engines (150-160 characters)</div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Meta Keywords</label>
                                    <textarea name="meta_keywords" class="form-control" rows="2" 
                                              placeholder="keyword1, keyword2, keyword3">{{ old('meta_keywords') }}</textarea>
                                    <div class="form-text">Comma-separated keywords</div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5">
                                                <i class="bx bx-save"></i> Save Brand
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5">
                                                <i class="bx bx-reset"></i> Reset Form
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('brands.index') }}" class="btn btn-light px-5">
                                                <i class="bx bx-x"></i> Cancel
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // Logo preview
        $('#logoInput').change(function() {
            if (this.files && this.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    $('#logoPreview').attr('src', e.target.result).show();
                }
                reader.readAsDataURL(this.files[0]);
            }
        });
        
        // Auto-generate slug from name
        $('input[name="name"]').on('blur', function() {
            if (!$('input[name="slug"]').val()) {
                const name = $(this).val();
                const slug = name.toLowerCase()
                    .replace(/[^a-z0-9\s-]/g, '')
                    .replace(/\s+/g, '-')
                    .replace(/-+/g, '-')
                    .trim();
                $('input[name="slug"]').val(slug);
            }
        });
        
        // Form validation
        $('#brandForm').validate({
            rules: {
                name: {
                    required: true,
                    minlength: 2,
                    maxlength: 255
                },
                website: {
                    url: true
                },
                contact_email: {
                    email: true
                },
                established_year: {
                    number: true,
                    min: 1800,
                    max: {{ date('Y') }}
                },
                order: {
                    number: true,
                    min: 0
                }
            },
            messages: {
                name: {
                    required: "Brand name is required",
                    minlength: "Brand name must be at least 2 characters"
                }
            },
            errorElement: 'span',
            errorPlacement: function (error, element) {
                error.addClass('invalid-feedback');
                element.closest('.mb-3').append(error);
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass('is-invalid');
            }
        });
    });
</script>
@endsection