// public/js/pos/ui/modals/payment-modal.js
class PaymentModal {
    constructor() {
        this.modal = null;
        this.stepHandlers = new Map();
    }
    
    init() {
        this.modal = new bootstrap.Modal(document.getElementById('paymentModal'));
        
        this.setupSteps();
        this.setupPaymentCards();
        this.setupEventListeners();
        
        // Subscribe to state changes
        posState.subscribe(this.onStateChange.bind(this));
    }
    
    setupSteps() {
        this.stepHandlers.set(1, this.showMethodSelection.bind(this));
        this.stepHandlers.set(2, this.showPaymentForm.bind(this));
        this.stepHandlers.set(3, this.showConfirmation.bind(this));
    }
    
    showMethodSelection() {
        this.renderPaymentMethods(posState.state.paymentMethods);
    }
    
    renderPaymentMethods(methods) {
        const container = document.querySelector('.payment-methods-grid');
        if (!container) return;
        
        container.innerHTML = methods.map(method => `
            <div class="payment-method-card ${method.code === posState.getSelectedMethod()?.code ? 'selected' : ''}"
                 data-method="${method.code}"
                 role="button"
                 tabindex="0">
                <div class="method-icon" style="background: ${this.getMethodColor(method.type)};">
                    <i class="bx bx-${method.icon}"></i>
                </div>
                <div class="method-info">
                    <div class="method-name">${method.name}</div>
                    <div class="method-desc">${method.description || ''}</div>
                </div>
                <div class="method-check"><i class="bx bx-check"></i></div>
            </div>
        `).join('');
    }
    
    getMethodColor(type) {
        const colors = {
            'cash': '#28a745',
            'mobile': '#00BFFF',
            'card': '#4169E1',
            'credit': '#FFA500',
            'bank': '#27AE60',
            'multiple': '#9B59B6'
        };
        return colors[type] || '#2a4365';
    }
    
    showPaymentForm() {
        const method = posState.getSelectedMethod();
        if (!method) return;
        
        // Load appropriate form template
        const formContainer = document.getElementById('paymentFormContainer');
        formContainer.innerHTML = this.getFormTemplate(method.code);
        
        // Initialize form
        this.initializeForm(method.code);
    }
    
    getFormTemplate(methodCode) {
        const templates = {
            'cash': this.getCashFormTemplate(),
            'mpesa': this.getMpesaFormTemplate(),
            'card': this.getCardFormTemplate(),
            'credit': this.getCreditFormTemplate(),
            'split': this.getSplitFormTemplate()
        };
        
        return templates[methodCode] || '<div>Form not available</div>';
    }
    
    getCashFormTemplate() {
        return `
            <div class="payment-form cash-form">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Total Amount</label>
                        <div class="input-group">
                            <span class="input-group-text">KES</span>
                            <input type="number" class="form-control" 
                                   id="cashTotal" value="${posState.getCartTotal().toFixed(2)}" readonly>
                        </div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label for="cashReceived" class="form-label">Cash Received</label>
                        <div class="input-group">
                            <span class="input-group-text">KES</span>
                            <input type="number" class="form-control" 
                                   id="cashReceived" value="${posState.getCartTotal().toFixed(2)}" 
                                   step="0.01" min="0">
                        </div>
                    </div>
                </div>
                <div id="cashChangeDisplay"></div>
            </div>
        `;
    }
    
    initializeForm(methodCode) {
        switch(methodCode) {
            case 'cash':
                this.initializeCashForm();
                break;
            case 'mpesa':
                this.initializeMpesaForm();
                break;
            // ... other forms
        }
    }
    
    initializeCashForm() {
        const cashReceivedInput = document.getElementById('cashReceived');
        if (cashReceivedInput) {
            cashReceivedInput.addEventListener('input', this.updateCashChange.bind(this));
            this.updateCashChange();
        }
    }
    
    updateCashChange() {
        const total = posState.getCartTotal();
        const received = parseFloat(document.getElementById('cashReceived')?.value || total);
        const change = received - total;
        
        const display = document.getElementById('cashChangeDisplay');
        if (!display) return;
        
        if (change >= 0) {
            display.innerHTML = `
                <div class="alert alert-success">
                    <strong>Change to give:</strong> KES ${change.toFixed(2)}
                </div>
            `;
        } else {
            display.innerHTML = `
                <div class="alert alert-danger">
                    <strong>Additional needed:</strong> KES ${Math.abs(change).toFixed(2)}
                </div>
            `;
        }
    }
    
    showConfirmation() {
        const method = posState.getSelectedMethod();
        const total = posState.getCartTotal();
        
        const container = document.getElementById('confirmationContent');
        if (container) {
            container.innerHTML = `
                <div class="confirmation-details">
                    <h5>Confirm Payment</h5>
                    <p><strong>Method:</strong> ${method.name}</p>
                    <p><strong>Amount:</strong> KES ${total.toFixed(2)}</p>
                    <p><strong>Customer:</strong> ${posState.state.currentCustomer?.name || 'Walk-in'}</p>
                    <p><strong>Items:</strong> ${posState.getCart().length} items</p>
                </div>
            `;
        }
    }
    
    setupPaymentCards() {
        document.addEventListener('click', (e) => {
            const card = e.target.closest('.payment-method-card');
            if (card) {
                const methodCode = card.dataset.method;
                this.selectPaymentMethod(methodCode);
            }
        });
    }
    
    selectPaymentMethod(methodCode) {
        try {
            const method = paymentManager.selectMethod(methodCode);
            this.updateStep(2);
        } catch (error) {
            POSUtils.showToast(error.message, 'error');
        }
    }
    
    setupEventListeners() {
        // Next/Prev buttons
        document.getElementById('nextStepBtn').addEventListener('click', () => {
            const currentStep = posState.getCurrentStep();
            if (currentStep === 2) {
                this.validateCurrentForm();
            }
            this.updateStep(currentStep + 1);
        });
        
        document.getElementById('prevStepBtn').addEventListener('click', () => {
            const currentStep = posState.getCurrentStep();
            this.updateStep(currentStep - 1);
        });
        
        // Complete payment
        document.getElementById('completePaymentBtn').addEventListener('click', async () => {
            await this.completePayment();
        });
    }
    
    updateStep(step) {
        if (step < 1 || step > 3) return;
        
        posState.updateUIState({ currentStep: step });
        
        // Update UI
        document.querySelectorAll('.payment-step').forEach((el, index) => {
            el.style.display = index + 1 === step ? 'block' : 'none';
        });
        
        // Update buttons visibility
        document.getElementById('prevStepBtn').style.display = step > 1 ? 'block' : 'none';
        document.getElementById('nextStepBtn').style.display = step < 3 ? 'block' : 'none';
        document.getElementById('completePaymentBtn').style.display = step === 3 ? 'block' : 'none';
        
        // Execute step handler
        const handler = this.stepHandlers.get(step);
        if (handler) handler();
    }
    
    validateCurrentForm() {
        const method = posState.getSelectedMethod();
        if (!method) return false;
        
        switch(method.code) {
            case 'cash':
                return this.validateCashForm();
            case 'mpesa':
                return this.validateMpesaForm();
            // ... other validations
        }
        
        return true;
    }
    
    validateCashForm() {
        const total = posState.getCartTotal();
        const received = parseFloat(document.getElementById('cashReceived')?.value || 0);
        
        if (received < total) {
            POSUtils.showToast(`Insufficient cash. Need KES ${(total - received).toFixed(2)} more`, 'error');
            return false;
        }
        
        return true;
    }
    
    async completePayment() {
        const method = posState.getSelectedMethod();
        const formData = this.collectFormData();
        
        try {
            const result = await paymentManager.processPayment({
                method: method.code,
                ...formData
            });
            
            this.showReceipt(result);
            this.modal.hide();
            
        } catch (error) {
            POSUtils.showToast(error.message, 'error');
        }
    }
    
    collectFormData() {
        const method = posState.getSelectedMethod();
        const data = {};
        
        switch(method.code) {
            case 'cash':
                data.cashReceived = parseFloat(document.getElementById('cashReceived')?.value || 0);
                break;
            case 'mpesa':
                data.phone = document.getElementById('mpesaPhone')?.value;
                break;
            // ... other methods
        }
        
        return data;
    }
    
    onStateChange(state) {
        // Update UI based on state changes
        this.updateStep(state.uiState.currentStep);
        
        if (state.uiState.isLoading) {
            this.showLoading();
        } else {
            this.hideLoading();
        }
    }
    
    show() {
        this.modal.show();
        this.updateStep(1);
    }
    
    showLoading() {
        document.getElementById('paymentSpinner')?.classList.remove('d-none');
    }
    
    hideLoading() {
        document.getElementById('paymentSpinner')?.classList.add('d-none');
    }
}

// Singleton instance
const paymentModal = new PaymentModal();