// public/js/pos/services/vat-service.js
// COMPLETE FIXED VERSION - VAT Calculation Service v4.0.0
// FIXED: Corrected price calculations for prices WITHOUT VAT storage

class VATService {
    constructor() {
        this.version = '4.0.0';
        console.log('💰 VAT Service initialized v' + this.version);
    }
    
    /**
     * Normalize product data from Laravel to consistent format
     * IMPORTANT: Prices from Laravel are WITHOUT VAT
     */
    normalizeProduct(product) {
        if (!product) {
            console.warn('⚠️ Product is null or undefined');
            return null;
        }
        
        console.log('🔄 Normalizing product:', { 
            id: product.id, 
            name: product.name,
            original_price: product.price,
            original_has_vat: product.has_vat 
        });
        
        // Extract VAT information based on Laravel Product model
        const hasVat = product.has_vat === true || 
                      product.has_vat === 1 || 
                      product.has_vat === '1' ||
                      product.has_vat === 'true';
        
        // Get tax rate from multiple possible sources
        let taxRate = 0;
        
        // Priority 1: Direct tax_rate field
        if (product.tax_rate !== undefined && product.tax_rate !== null) {
            taxRate = parseFloat(product.tax_rate);
            console.log('📊 Using direct tax_rate:', taxRate);
        }
        // Priority 2: From tax_info array
        else if (product.tax_info && typeof product.tax_info === 'object' && product.tax_info.rate) {
            taxRate = parseFloat(product.tax_info.rate);
            console.log('📊 Using tax_info rate:', taxRate);
        }
        // Priority 3: From tax relationship
        else if (product.tax && typeof product.tax === 'object' && product.tax.rate) {
            taxRate = parseFloat(product.tax.rate);
            console.log('📊 Using tax relationship rate:', taxRate);
        }
        // Priority 4: From getTaxRateAttribute (if accessible)
        else if (product.taxRate !== undefined && product.taxRate !== null) {
            taxRate = parseFloat(product.taxRate);
            console.log('📊 Using taxRate attribute:', taxRate);
        }
        
        // FIXED: Product is vatable only if it has VAT AND tax rate > 0
        const isVatable = hasVat && taxRate > 0;
        
        // Handle price values - IMPORTANT: Prices from Laravel are WITHOUT VAT
        // Use price_without_tax if available, otherwise use price/sale_price
        const priceWithoutVAT = parseFloat(product.price_without_tax) || 
                               parseFloat(product.price_ex_tax) || 
                               parseFloat(product.price_excluding_tax) ||
                               parseFloat(product.final_price) || 
                               parseFloat(product.sale_price) || 
                               parseFloat(product.price) || 0;
        
        // Calculate price WITH VAT for display
        const priceWithVAT = this.calculatePriceWithVAT(priceWithoutVAT, taxRate, isVatable);
        
        // Get quantity
        const quantity = parseInt(product.quantity) || 
                        parseInt(product.qty) || 
                        1;
        
        const normalizedProduct = {
            // Core product info
            id: product.id || product.product_id,
            sku: product.sku || '',
            barcode: product.barcode || '',
            name: product.name || product.product_name || 'Unknown Product',
            
            // FIXED: Store price WITHOUT VAT as price_ex_tax
            price: priceWithVAT, // For display in UI (with VAT if applicable)
            price_ex_tax: priceWithoutVAT, // For calculations (without VAT)
            
            // Cost price
            cost_price: parseFloat(product.cost_price) || 0,
            
            // VAT info - FIXED LOGIC
            has_vat: hasVat,
            is_vatable: isVatable, // Only true if hasVat AND taxRate > 0
            tax_rate: taxRate,
            
            // Quantity
            quantity: quantity,
            qty: quantity,
            
            // Stock info
            stock: parseInt(product.stock) || 0,
            available_stock: parseInt(product.available_stock) || parseInt(product.stock) || 0,
            
            // Additional fields
            category_id: product.category_id,
            unit_id: product.unit_id,
            brand_id: product.brand_id,
            
            // Original data for reference
            _original: product
        };
        
        console.log('✅ Normalized product:', {
            name: normalizedProduct.name,
            priceWithoutVAT: normalizedProduct.price_ex_tax,
            priceWithVAT: normalizedProduct.price,
            hasVat: hasVat,
            isVatable: isVatable,
            taxRate: taxRate
        });
        
        return normalizedProduct;
    }
    
    /**
     * Calculate price WITH VAT
     */
    calculatePriceWithVAT(priceWithoutVAT, vatRate, isVatable) {
        if (!isVatable || vatRate <= 0) {
            return priceWithoutVAT;
        }
        
        // Calculate price including VAT: price * (1 + vatRate/100)
        return priceWithoutVAT * (1 + (vatRate / 100));
    }
    
    /**
     * Calculate price without VAT (reverse calculation)
     */
    calculatePriceWithoutVAT(priceWithVAT, vatRate) {
        if (vatRate <= 0) {
            return priceWithVAT;
        }
        
        // Calculate price excluding VAT: price / (1 + vatRate/100)
        return priceWithVAT / (1 + (vatRate / 100));
    }
    
    /**
     * Calculate VAT for a single item based on customer status
     */
    calculateItemVAT(item, customerStatus = 'vatable') {
        // Validate inputs
        if (!item) {
            console.warn('⚠️ Item is null or undefined in calculateItemVAT');
            return this.getEmptyItemVATData();
        }
        
        // Normalize product data
        const normalizedItem = this.normalizeProduct(item);
        
        if (!normalizedItem) {
            console.error('❌ Failed to normalize product item');
            return this.getEmptyItemVATData();
        }
        
        const quantity = normalizedItem.quantity;
        const priceExcludingVAT = normalizedItem.price_ex_tax; // Price WITHOUT VAT
        const itemTotalExcludingVAT = priceExcludingVAT * quantity;
        
        // Get product VAT information
        const productTaxRate = normalizedItem.tax_rate || 0;
        const isProductVatable = normalizedItem.is_vatable || false;
        const hasVAT = normalizedItem.has_vat || false;
        
        console.log('🔍 VAT Calculation for item:', {
            name: normalizedItem.name,
            hasVAT: hasVAT,
            isProductVatable: isProductVatable,
            productTaxRate: productTaxRate,
            customerStatus: customerStatus,
            priceExcludingVAT: priceExcludingVAT,
            quantity: quantity,
            totalExcludingVAT: itemTotalExcludingVAT
        });
        
        // Apply customer VAT status rules
        let vatRate = 0;
        let vatAmount = 0;
        let vatType = 'none';
        let isVATApplicable = false;
        
        switch (customerStatus) {
            case 'exempted':
                // VAT exempted customer - NO VAT on any items
                vatRate = 0;
                vatAmount = 0;
                vatType = 'exempted';
                isVATApplicable = false;
                console.log('✅ Customer exempted - No VAT');
                break;
                
            case 'zero_rated':
                // Zero-rated customer - taxable at 0% rate ONLY IF product is VATable
                if (isProductVatable) {
                    vatRate = 0;
                    vatAmount = 0;
                    vatType = 'zero_rated';
                    isVATApplicable = true;
                    console.log('✅ Zero-rated customer - 0% VAT on VATable items');
                } else {
                    vatRate = 0;
                    vatAmount = 0;
                    vatType = 'non_vatable';
                    isVATApplicable = false;
                    console.log('✅ Zero-rated customer - Non-VAT item remains non-taxable');
                }
                break;
                
            case 'vatable':
            default:
                // VATable customer - apply VAT ONLY IF product is VATable AND has tax rate > 0
                if (isProductVatable && productTaxRate > 0) {
                    vatRate = productTaxRate;
                    vatAmount = (itemTotalExcludingVAT * vatRate) / 100;
                    vatType = 'standard';
                    isVATApplicable = true;
                    console.log('✅ VATable customer - VAT applied:', vatAmount);
                } else {
                    vatRate = 0;
                    vatAmount = 0;
                    vatType = 'non_vatable';
                    isVATApplicable = false;
                    console.log('✅ VATable customer - No VAT applicable');
                }
                break;
        }
        
        const totalIncludingVAT = itemTotalExcludingVAT + vatAmount;
        
        const result = {
            // Item details
            itemId: normalizedItem.id,
            productName: normalizedItem.name,
            quantity: quantity,
            priceExcludingVAT: priceExcludingVAT,
            
            // VAT details
            vatRate: vatRate,
            vatAmount: vatAmount,
            totalExcludingVAT: itemTotalExcludingVAT,
            totalIncludingVAT: totalIncludingVAT,
            
            // Status info
            vatType: vatType,
            isVATApplicable: isVATApplicable,
            isProductVatable: isProductVatable,
            customerVatStatus: customerStatus,
            isExempted: customerStatus === 'exempted',
            isZeroRated: customerStatus === 'zero_rated',
            isVatableCustomer: customerStatus === 'vatable',
            
            // Original data
            originalItem: normalizedItem._original,
            
            // Metadata
            calculatedAt: new Date().toISOString()
        };
        
        console.log('📊 Item VAT Result:', {
            name: result.productName,
            priceExcludingVAT: result.priceExcludingVAT,
            vatAmount: result.vatAmount,
            totalExcludingVAT: result.totalExcludingVAT,
            totalIncludingVAT: result.totalIncludingVAT
        });
        
        return result;
    }
    
    /**
     * Calculate VAT for entire cart with customer status
     */
    calculateCartVAT(cartItems, customer = null, discount = 0) {
        console.log('🔄 Calculating cart VAT:', {
            itemCount: cartItems?.length || 0,
            customer: customer?.name || 'Walk-in',
            customerStatus: customer?.vat_status || 'vatable',
            discount: discount
        });
        
        // Validate inputs
        if (!Array.isArray(cartItems) || cartItems.length === 0) {
            console.log('🛒 Empty cart - returning empty VAT data');
            return this.getEmptyCartVATData(customer);
        }
        
        // Get customer status
        const customerStatus = customer?.vat_status || 'vatable';
        const customerName = customer?.name || 'Walk-in Customer';
        const isExempted = customerStatus === 'exempted';
        const isZeroRated = customerStatus === 'zero_rated';
        const isVatable = customerStatus === 'vatable' || !customer;
        
        // Initialize totals
        let subtotalExcludingVAT = 0;
        let totalVAT = 0;
        let taxableAmount = 0;
        let nonTaxableAmount = 0;
        let exemptedAmount = 0;
        let zeroRatedAmount = 0;
        const vatByRate = {};
        
        // Track item types
        let vatableItemsCount = 0;
        let nonVatableItemsCount = 0;
        
        // Process each item
        const itemsWithVAT = [];
        
        for (let i = 0; i < cartItems.length; i++) {
            const item = cartItems[i];
            console.log(`🛒 Processing item ${i + 1}/${cartItems.length}:`, {
                name: item?.name || item?.product_name,
                has_vat: item?.has_vat,
                tax_rate: item?.tax_rate
            });
            
            try {
                const itemVAT = this.calculateItemVAT(item, customerStatus);
                
                // Track item types
                if (itemVAT.isProductVatable) {
                    vatableItemsCount++;
                } else {
                    nonVatableItemsCount++;
                }
                
                // Accumulate totals (using amounts EXCLUDING VAT)
                subtotalExcludingVAT += itemVAT.totalExcludingVAT;
                totalVAT += itemVAT.vatAmount;
                
                // Categorize amounts
                switch (itemVAT.vatType) {
                    case 'exempted':
                        exemptedAmount += itemVAT.totalExcludingVAT;
                        nonTaxableAmount += itemVAT.totalExcludingVAT;
                        console.log('📝 Item categorized as exempted');
                        break;
                        
                    case 'zero_rated':
                        zeroRatedAmount += itemVAT.totalExcludingVAT;
                        taxableAmount += itemVAT.totalExcludingVAT;
                        
                        // Track zero-rated items
                        if (!vatByRate[0]) {
                            vatByRate[0] = {
                                rate: 0,
                                amount: 0,
                                taxableAmount: 0,
                                type: 'zero_rated',
                                description: 'Zero-rated VAT'
                            };
                        }
                        vatByRate[0].taxableAmount += itemVAT.totalExcludingVAT;
                        console.log('📝 Item categorized as zero-rated');
                        break;
                        
                    case 'standard':
                        taxableAmount += itemVAT.totalExcludingVAT;
                        
                        // Group by VAT rate
                        const rateKey = itemVAT.vatRate;
                        if (!vatByRate[rateKey]) {
                            vatByRate[rateKey] = {
                                rate: itemVAT.vatRate,
                                amount: 0,
                                taxableAmount: 0,
                                type: 'standard',
                                description: `VAT ${itemVAT.vatRate}%`
                            };
                        }
                        vatByRate[rateKey].amount += itemVAT.vatAmount;
                        vatByRate[rateKey].taxableAmount += itemVAT.totalExcludingVAT;
                        console.log(`📝 Item categorized as standard VAT ${itemVAT.vatRate}%`);
                        break;
                        
                    case 'non_vatable':
                    default:
                        nonTaxableAmount += itemVAT.totalExcludingVAT;
                        console.log('📝 Item categorized as non-vatable');
                        break;
                }
                
                itemsWithVAT.push(itemVAT);
                
            } catch (error) {
                console.error(`❌ Error processing item ${i}:`, error, item);
            }
        }
        
        console.log('📊 Cart Totals Before Discount:', {
            subtotalExcludingVAT: subtotalExcludingVAT,
            totalVAT: totalVAT,
            taxableAmount: taxableAmount,
            nonTaxableAmount: nonTaxableAmount,
            vatableItemsCount: vatableItemsCount,
            nonVatableItemsCount: nonVatableItemsCount
        });
        
        // Apply discount proportionally (discount applies to subtotal excluding VAT)
        let discountedVAT = totalVAT;
        let discountedTaxableAmount = taxableAmount;
        let discountedNonTaxableAmount = nonTaxableAmount;
        let discountedExemptedAmount = exemptedAmount;
        let discountedZeroRatedAmount = zeroRatedAmount;
        
        if (discount > 0 && subtotalExcludingVAT > 0) {
            const discountRatio = Math.min(discount / subtotalExcludingVAT, 1);
            console.log(`📉 Applying discount: ${discount} (ratio: ${discountRatio.toFixed(4)})`);
            
            // Apply discount proportionally to all amounts
            discountedVAT = totalVAT * (1 - discountRatio);
            discountedTaxableAmount = taxableAmount * (1 - discountRatio);
            discountedNonTaxableAmount = nonTaxableAmount * (1 - discountRatio);
            discountedExemptedAmount = exemptedAmount * (1 - discountRatio);
            discountedZeroRatedAmount = zeroRatedAmount * (1 - discountRatio);
            
            // Also adjust vatByRate amounts proportionally
            Object.keys(vatByRate).forEach(rate => {
                vatByRate[rate].amount *= (1 - discountRatio);
                vatByRate[rate].taxableAmount *= (1 - discountRatio);
            });
        }
        
        const discountedSubtotalExcludingVAT = Math.max(0, subtotalExcludingVAT - discount);
        const totalWithVAT = discountedSubtotalExcludingVAT + discountedVAT;
        
        // VAT notes
        const vatNotes = this.generateVATNotes(
            customerStatus, 
            isExempted, 
            isZeroRated, 
            discountedVAT,
            vatableItemsCount,
            nonVatableItemsCount
        );
        
        const result = {
            // Customer info
            customer: {
                id: customer?.id || null,
                name: customerName,
                vatStatus: customerStatus
            },
            
            // Items with VAT breakdown
            items: itemsWithVAT,
            
            // Totals
            subtotal: subtotalExcludingVAT,
            discount: discount,
            discountedSubtotal: discountedSubtotalExcludingVAT,
            
            // VAT breakdown
            taxableAmount: discountedTaxableAmount,
            nonTaxableAmount: discountedNonTaxableAmount,
            exemptedAmount: discountedExemptedAmount,
            zeroRatedAmount: discountedZeroRatedAmount,
            totalVAT: discountedVAT,
            totalWithVAT: totalWithVAT,
            
            // VAT rates breakdown
            vatByRate: vatByRate,
            
            // Status flags
            customerVatStatus: customerStatus,
            isVatApplicable: !isExempted,
            isZeroRated: isZeroRated,
            isVatable: isVatable,
            isExempted: isExempted,
            hasVAT: discountedVAT > 0,
            
            // Item counts
            vatableItemsCount: vatableItemsCount,
            nonVatableItemsCount: nonVatableItemsCount,
            totalItemsCount: cartItems.length,
            
            // Notes and metadata
            vatNotes: vatNotes,
            itemsCount: cartItems.length,
            calculatedAt: new Date().toISOString()
        };
        
        console.log('✅ Final Cart VAT Result:', {
            subtotal: result.subtotal,
            totalVAT: result.totalVAT,
            discount: result.discount,
            totalWithVAT: result.totalWithVAT,
            taxableAmount: result.taxableAmount,
            nonTaxableAmount: result.nonTaxableAmount,
            hasVAT: result.hasVAT
        });
        
        return result;
    }
    
    /**
     * Generate VAT notes based on customer status
     */
    generateVATNotes(customerStatus, isExempted, isZeroRated, totalVAT = 0, vatableItemsCount = 0, nonVatableItemsCount = 0) {
        const notes = [];
        
        if (isExempted) {
            notes.push('VAT EXEMPTED CUSTOMER');
            notes.push('No VAT applied to any items');
            notes.push('Tax Exemption Certificate on file');
        } else if (isZeroRated) {
            notes.push('ZERO-RATED CUSTOMER');
            notes.push('VAT applicable at 0% rate on VATable items only');
            if (vatableItemsCount > 0) {
                notes.push(`${vatableItemsCount} VATable item(s) at 0% rate`);
            }
            if (nonVatableItemsCount > 0) {
                notes.push(`${nonVatableItemsCount} non-VAT item(s) remain non-taxable`);
            }
        } else if (totalVAT > 0) {
            notes.push('STANDARD VAT CUSTOMER');
            notes.push(`VAT applied to ${vatableItemsCount} VATable item(s)`);
            notes.push(`VAT Amount: KES ${totalVAT.toFixed(2)}`);
            if (nonVatableItemsCount > 0) {
                notes.push(`${nonVatableItemsCount} non-VAT item(s) not taxed`);
            }
        } else {
            notes.push('NO VAT APPLICABLE');
            if (vatableItemsCount > 0) {
                notes.push(`${vatableItemsCount} item(s) are VATable but at 0% rate`);
            } else {
                notes.push('Items in cart are not VATable');
            }
        }
        
        notes.push('VAT compliant invoice');
        notes.push('Valid for tax purposes');
        
        return notes;
    }
    
    /**
     * Get VAT status description
     */
    getVATStatusDescription(status) {
        const descriptions = {
            'exempted': 'VAT Exempted - No VAT applied',
            'zero_rated': 'Zero-Rated - 0% VAT rate on VATable items only',
            'vatable': 'VATable - Standard VAT rates apply to VATable items'
        };
        return descriptions[status] || 'VATable - Standard VAT rates apply';
    }
    
    /**
     * Check if product has VAT (based on Laravel model)
     */
    checkProductHasVAT(product) {
        if (!product) {
            console.warn('⚠️ Product is null in checkProductHasVAT');
            return false;
        }
        
        const normalized = this.normalizeProduct(product);
        if (!normalized) return false;
        
        // FIXED: Product has VAT only if has_vat AND tax_rate > 0
        return normalized.has_vat && normalized.tax_rate > 0;
    }
    
    /**
     * Get product tax rate (based on Laravel model)
     */
    getProductTaxRate(product) {
        if (!product) {
            console.warn('⚠️ Product is null in getProductTaxRate');
            return 0;
        }
        
        let taxRate = 0;
        
        // Priority 1: Direct tax_rate field
        if (product.tax_rate !== undefined && product.tax_rate !== null) {
            taxRate = parseFloat(product.tax_rate);
            if (!isNaN(taxRate)) {
                console.log(`📊 Using direct tax_rate: ${taxRate}%`);
                return taxRate;
            }
        }
        
        // Priority 2: From tax_info array
        if (product.tax_info && typeof product.tax_info === 'object') {
            if (product.tax_info.rate !== undefined && product.tax_info.rate !== null) {
                taxRate = parseFloat(product.tax_info.rate);
                if (!isNaN(taxRate)) {
                    console.log(`📊 Using tax_info rate: ${taxRate}%`);
                    return taxRate;
                }
            }
        }
        
        // Priority 3: From tax relationship
        if (product.tax && typeof product.tax === 'object') {
            if (product.tax.rate !== undefined && product.tax.rate !== null) {
                taxRate = parseFloat(product.tax.rate);
                if (!isNaN(taxRate)) {
                    console.log(`📊 Using tax relationship rate: ${taxRate}%`);
                    return taxRate;
                }
            }
        }
        
        // Priority 4: From getTaxRateAttribute (if accessible)
        if (product.taxRate !== undefined && product.taxRate !== null) {
            taxRate = parseFloat(product.taxRate);
            if (!isNaN(taxRate)) {
                console.log(`📊 Using taxRate attribute: ${taxRate}%`);
                return taxRate;
            }
        }
        
        // Check if product has VAT but no rate specified
        const hasVat = product.has_vat === true || 
                      product.has_vat === 1 || 
                      product.has_vat === '1' ||
                      product.has_vat === 'true';
        
        if (hasVat) {
            // Default VAT rate (Kenya standard)
            console.log('📊 Using default VAT rate: 16%');
            return 16.00;
        }
        
        console.log('📊 No VAT rate found - returning 0%');
        return 0;
    }
    
    /**
     * Validate cart items for VAT calculation
     */
    validateCartItems(cartItems) {
        const errors = [];
        const validatedItems = [];
        
        if (!Array.isArray(cartItems)) {
            errors.push('Cart items must be an array');
            return {
                isValid: false,
                errors: errors,
                items: []
            };
        }
        
        for (let i = 0; i < cartItems.length; i++) {
            const item = cartItems[i];
            
            try {
                if (!item) {
                    errors.push(`Item at index ${i} is null or undefined`);
                    continue;
                }
                
                // Validate required fields
                if (!item.id && !item.product_id) {
                    errors.push(`Item at index ${i} has no ID`);
                }
                
                if (!item.name && !item.product_name) {
                    errors.push(`Item at index ${i} has no name`);
                }
                
                // Validate price
                const normalized = this.normalizeProduct(item);
                const priceExcludingVAT = normalized?.price_ex_tax || 0;
                
                if (isNaN(priceExcludingVAT)) {
                    errors.push(`Item "${item.name || item.product_name}" has invalid price`);
                }
                
                if (priceExcludingVAT < 0) {
                    errors.push(`Item "${item.name || item.product_name}" has negative price: ${priceExcludingVAT}`);
                }
                
                // Validate quantity
                const quantity = parseInt(item.quantity) || 
                                parseInt(item.qty) || 
                                1;
                
                if (isNaN(quantity) || quantity <= 0) {
                    errors.push(`Item "${item.name || item.product_name}" has invalid quantity: ${quantity}`);
                }
                
                // Validate VAT data consistency
                if (!normalized) {
                    errors.push(`Failed to normalize item "${item.name || item.product_name}"`);
                    continue;
                }
                
                const isVatable = normalized.is_vatable;
                const taxRate = normalized.tax_rate;
                
                if (isVatable && taxRate === 0) {
                    errors.push('VATable item must have a tax rate > 0');
                }
                
                validatedItems.push({
                    ...item,
                    _normalized: normalized,
                    _vatValid: true
                });
                
            } catch (error) {
                errors.push(`Error validating item at index ${i}: ${error.message}`);
            }
        }
        
        return {
            isValid: errors.length === 0,
            errors: errors,
            items: validatedItems
        };
    }
    
    /**
     * Empty cart VAT data
     */
    getEmptyCartVATData(customer = null) {
        const customerStatus = customer?.vat_status || 'vatable';
        const isExempted = customerStatus === 'exempted';
        const isZeroRated = customerStatus === 'zero_rated';
        
        const emptyData = {
            customer: {
                id: customer?.id || null,
                name: customer?.name || 'Walk-in Customer',
                vatStatus: customerStatus
            },
            items: [],
            subtotal: 0,
            discount: 0,
            discountedSubtotal: 0,
            taxableAmount: 0,
            nonTaxableAmount: 0,
            exemptedAmount: 0,
            zeroRatedAmount: 0,
            totalVAT: 0,
            totalWithVAT: 0,
            vatByRate: {},
            customerVatStatus: customerStatus,
            isVatApplicable: !isExempted,
            isZeroRated: isZeroRated,
            isVatable: customerStatus === 'vatable' || !customer,
            isExempted: isExempted,
            hasVAT: false,
            vatableItemsCount: 0,
            nonVatableItemsCount: 0,
            totalItemsCount: 0,
            vatNotes: ['No items in cart'],
            itemsCount: 0,
            calculatedAt: new Date().toISOString()
        };
        
        console.log('🔄 Returning empty cart VAT data:', emptyData);
        return emptyData;
    }
    
    /**
     * Empty item VAT data
     */
    getEmptyItemVATData() {
        return {
            itemId: null,
            productName: 'Unknown',
            quantity: 1,
            priceExcludingVAT: 0,
            vatRate: 0,
            vatAmount: 0,
            totalExcludingVAT: 0,
            totalIncludingVAT: 0,
            vatType: 'none',
            isVATApplicable: false,
            isProductVatable: false,
            customerVatStatus: 'vatable',
            isExempted: false,
            isZeroRated: false,
            isVatableCustomer: true,
            calculatedAt: new Date().toISOString()
        };
    }
    
    /**
     * Format VAT amount for display
     */
    formatVATDisplay(vatData) {
        if (!vatData) {
            console.warn('⚠️ vatData is null in formatVATDisplay');
            return {
                status: 'error',
                message: 'ERROR',
                color: 'danger',
                icon: 'bx-error',
                amount: 'KES 0.00',
                description: 'VAT data error',
                showVATSection: false
            };
        }
        
        if (vatData.isExempted) {
            return {
                status: 'exempted',
                message: 'VAT EXEMPTED',
                color: 'danger',
                icon: 'bx-shield-x',
                amount: 'KES 0.00',
                description: 'No VAT applied to any items',
                showVATSection: false
            };
        } else if (vatData.isZeroRated) {
            return {
                status: 'zero_rated',
                message: 'ZERO-RATED',
                color: 'warning',
                icon: 'bx-shield-alt-2',
                amount: 'KES 0.00',
                description: 'VAT applicable at 0% rate on VATable items',
                showVATSection: true
            };
        } else if (vatData.hasVAT) {
            return {
                status: 'vatable',
                message: 'VAT APPLIED',
                color: 'success',
                icon: 'bx-receipt',
                amount: `KES ${vatData.totalVAT.toFixed(2)}`,
                description: `VAT included in total (${vatData.vatableItemsCount} VATable item(s))`,
                showVATSection: true
            };
        } else {
            return {
                status: 'no_vat',
                message: 'NO VAT',
                color: 'secondary',
                icon: 'bx-minus-circle',
                amount: 'KES 0.00',
                description: 'No VAT applicable to items in cart',
                showVATSection: false
            };
        }
    }
    
    /**
     * Quick VAT check - returns true if any item in cart is VATable
     */
    hasVATableItems(cartItems) {
        if (!Array.isArray(cartItems) || cartItems.length === 0) {
            return false;
        }
        
        for (let i = 0; i < cartItems.length; i++) {
            const item = cartItems[i];
            const normalized = this.normalizeProduct(item);
            
            // FIXED: Check both has_vat AND tax_rate > 0
            if (normalized && normalized.has_vat && normalized.tax_rate > 0) {
                console.log(`✅ Found VATable item: ${normalized.name}`);
                return true;
            }
        }
        
        console.log('❌ No VATable items found');
        return false;
    }
    
    /**
     * Get VAT summary for display
     */
    getVATSummary(vatData) {
        if (!vatData || vatData.itemsCount === 0) {
            return {
                showVAT: false,
                message: 'No items in cart',
                vatAmount: 0,
                taxableAmount: 0,
                nonTaxableAmount: 0
            };
        }
        
        return {
            showVAT: vatData.hasVAT || vatData.isExempted || vatData.isZeroRated,
            message: vatData.hasVAT ? 
                `VAT: KES ${vatData.totalVAT.toFixed(2)}` : 
                (vatData.isExempted ? 'VAT Exempted' : 'No VAT applicable'),
            vatAmount: vatData.totalVAT,
            taxableAmount: vatData.taxableAmount,
            nonTaxableAmount: vatData.nonTaxableAmount,
            vatByRate: vatData.vatByRate
        };
    }
    
    /**
     * Validate item VAT data
     */
    validateItemVATData(item) {
        const errors = [];
        
        if (!item) {
            errors.push('Item is required');
            return { 
                isValid: false, 
                errors: errors 
            };
        }
        
        // Check for required fields
        if (!item.name && !item.product_name) {
            errors.push('Item name is required');
        }
        
        const normalized = this.normalizeProduct(item);
        
        if (!normalized) {
            errors.push('Failed to normalize product data');
            return {
                isValid: false,
                errors: errors
            };
        }
        
        const priceExcludingVAT = normalized.price_ex_tax || 0;
        
        if (priceExcludingVAT <= 0) {
            errors.push('Item price must be greater than 0');
        }
        
        const isVatable = normalized.is_vatable;
        const taxRate = normalized.tax_rate;
        
        if (isVatable && taxRate === 0) {
            errors.push('VATable item must have a tax rate > 0');
        }
        
        if (!isVatable && taxRate > 0) {
            errors.push('Non-VATable item should not have a tax rate');
        }
        
        return {
            isValid: errors.length === 0,
            errors: errors,
            isVatable: isVatable,
            taxRate: taxRate,
            normalizedItem: normalized
        };
    }
    
    /**
     * Calculate VAT for a specific amount (for manual calculations)
     */
    calculateVATForAmount(amount, vatRate = 16, isVatable = true) {
        if (!isVatable || vatRate <= 0) {
            return {
                amount: amount,
                vatRate: 0,
                vatAmount: 0,
                totalWithVAT: amount
            };
        }
        
        const vatAmount = (amount * vatRate) / 100;
        
        return {
            amount: amount,
            vatRate: vatRate,
            vatAmount: vatAmount,
            totalWithVAT: amount + vatAmount
        };
    }
    
    /**
     * Calculate amount excluding VAT
     */
    calculateAmountExcludingVAT(totalAmount, vatRate = 16) {
        if (vatRate <= 0) {
            return {
                totalAmount: totalAmount,
                vatRate: 0,
                amountExcludingVAT: totalAmount,
                vatAmount: 0
            };
        }
        
        const amountExcludingVAT = totalAmount / (1 + (vatRate / 100));
        const vatAmount = totalAmount - amountExcludingVAT;
        
        return {
            totalAmount: totalAmount,
            vatRate: vatRate,
            amountExcludingVAT: amountExcludingVAT,
            vatAmount: vatAmount
        };
    }
    
    /**
     * Debug function to check product VAT status
     */
    debugProductVAT(product) {
        console.log('🔍 Debug Product VAT:', {
            original: product,
            normalized: this.normalizeProduct(product),
            hasVAT: this.checkProductHasVAT(product),
            taxRate: this.getProductTaxRate(product),
            validation: this.validateItemVATData(product)
        });
    }
    
    /**
     * Test function to verify VAT calculations
     */
    testVATCalculations() {
        console.log('🧪 Testing VAT Calculations...');
        
        // Test case 1: Non-VAT product
        const nonVATProduct = {
            id: 1,
            name: 'Non-VAT Item',
            sale_price: 100,
            final_price: 100,
            has_vat: false,
            tax_rate: 0
        };
        
        // Test case 2: VAT product
        const vatProduct = {
            id: 2,
            name: 'VAT Item',
            sale_price: 100,
            final_price: 100,
            has_vat: true,
            tax_rate: 16
        };
        
        const testCases = [nonVATProduct, vatProduct];
        
        testCases.forEach((product, index) => {
            console.log(`\n📦 Test Case ${index + 1}: ${product.name}`);
            this.debugProductVAT(product);
            
            const vatResult = this.calculateItemVAT(product, 'vatable');
            console.log('📊 VAT Result:', {
                priceExcludingVAT: vatResult.priceExcludingVAT,
                vatAmount: vatResult.vatAmount,
                totalIncludingVAT: vatResult.totalIncludingVAT,
                vatType: vatResult.vatType
            });
        });
        
        // Test cart calculation
        const testCart = [nonVATProduct, vatProduct];
        const cartVAT = this.calculateCartVAT(testCart, null, 0);
        
        console.log('\n🛒 Cart VAT Test Result:', {
            subtotal: cartVAT.subtotal,
            totalVAT: cartVAT.totalVAT,
            totalWithVAT: cartVAT.totalWithVAT,
            hasVAT: cartVAT.hasVAT
        });
    }
}

// Create global instance
window.VATService = new VATService();

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = VATService;
}

console.log('✅ VAT Service loaded v4.0.0 - FIXED: Prices stored WITHOUT VAT');

// Add helper function for product preparation
window.prepareProductForCart = function(product) {
    console.log('🔄 Preparing product for cart:', product);
    
    if (!product) {
        console.error('❌ Product is null or undefined');
        return null;
    }
    
    try {
        // Use VATService to normalize the product
        const normalized = window.VATService.normalizeProduct(product);
        
        if (!normalized) {
            console.error('❌ Failed to normalize product');
            return null;
        }
        
        // Create cart-ready product
        const cartProduct = {
            id: normalized.id,
            sku: normalized.sku,
            barcode: normalized.barcode,
            name: normalized.name,
            price: normalized.price, // Display price (with VAT if applicable)
            price_ex_tax: normalized.price_ex_tax, // Price without VAT for calculations
            cost_price: normalized.cost_price,
            
            // VAT information
            has_vat: normalized.has_vat,
            is_vatable: normalized.is_vatable,
            tax_rate: normalized.tax_rate,
            
            // Quantity
            quantity: normalized.quantity,
            qty: normalized.quantity,
            
            // Stock info
            stock: normalized.stock,
            available_stock: normalized.available_stock,
            
            // Additional info
            category_id: normalized.category_id,
            unit_id: normalized.unit_id,
            brand_id: normalized.brand_id,
            
            // Keep original for reference
            _original: normalized._original
        };
        
        console.log('✅ Prepared cart product:', cartProduct);
        return cartProduct;
        
    } catch (error) {
        console.error('❌ Error preparing product for cart:', error);
        return null;
    }
};

// Add global debug function
window.debugVATCalculation = function() {
    if (window.VATService && window.VATService.testVATCalculations) {
        window.VATService.testVATCalculations();
    } else {
        console.error('VATService not available');
    }
};