// =========================================================================
// POS Utilities Module - Production Grade v2.1.0
// =========================================================================
// Features: Complete error handling, robust methods, proper exports
// Now includes number generation utilities
// =========================================================================

(function() {
    'use strict';
    
    class POSUtils {
        constructor() {
            this.version = '2.1.0';
            this.initialized = false;
            this.debounceTimers = new Map();
            this.throttleFlags = new Map();
            this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
            this.toastContainer = null;
            this.numberManager = new NumberManager();
            
            // Auto-initialize
            this._autoInit();
        }
        
        _autoInit() {
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => {
                    this.init();
                });
            } else {
                this.init();
            }
        }
        
        init() {
            if (this.initialized) return this;
            
            console.log('🔄 Initializing POS Utils v' + this.version);
            
            try {
                this.createToastContainer();
                this.createLoadingOverlay();
                this.setupGlobalErrorHandling();
                this.initialized = true;
                console.log('✅ POS Utils module loaded');
            } catch (error) {
                console.error('Failed to initialize POS Utils:', error);
            }
            
            return this;
        }
        
        // ==================== NUMBER MANAGEMENT ====================
        
        /**
         * Generate invoice and receipt numbers
         */
        async generateNumbers() {
            try {
                const response = await this.fetchJson('/api/numbers/generate');
                if (response.success) {
                    return response.numbers;
                }
                return this.generateFallbackNumbers();
            } catch (error) {
                console.warn('Failed to generate numbers from server, using fallback');
                return this.generateFallbackNumbers();
            }
        }
        
        /**
         * Generate fallback numbers
         */
        generateFallbackNumbers() {
            const date = new Date();
            const dateStr = date.toISOString().slice(0,10).replace(/-/g,'');
            
            // Try to get sequence from localStorage
            const invoiceSequence = this.getNextSequence('invoice');
            const receiptSequence = this.getNextSequence('receipt');
            
            return {
                invoice: `INV-${dateStr}-${invoiceSequence.toString().padStart(3, '0')}`,
                receipt: `RCPT-${dateStr}-${receiptSequence.toString().padStart(3, '0')}`,
                date: dateStr,
                timestamp: date.toISOString(),
                is_fallback: true
            };
        }
        
        /**
         * Get next sequence from localStorage
         */
        getNextSequence(type) {
            const key = `pos_sequence_${type}`;
            const dateKey = `pos_sequence_date_${type}`;
            const today = new Date().toDateString();
            
            // Reset sequence if date changed
            const lastDate = localStorage.getItem(dateKey);
            if (lastDate !== today) {
                localStorage.setItem(dateKey, today);
                localStorage.setItem(key, '1');
                return 1;
            }
            
            // Increment sequence
            const current = parseInt(localStorage.getItem(key) || '0') + 1;
            localStorage.setItem(key, current.toString());
            return current;
        }
        
        /**
         * Validate a number
         */
        validateNumber(number, type = 'invoice') {
            const patterns = {
                invoice: /^INV-\d{8}-\d{3}$/,
                receipt: /^RCPT-\d{8}-\d{3}$/,
                refund: /^REF-\d{8}-\d{3}$/,
                credit_note: /^CN-\d{8}-\d{3}$/
            };
            
            const pattern = patterns[type] || patterns.invoice;
            return pattern.test(number);
        }
        
        /**
         * Parse a number to get its components
         */
        parseNumber(number) {
            const parts = number.split('-');
            
            if (parts.length >= 3) {
                const prefix = parts[0];
                const date = parts[1];
                const sequence = parseInt(parts[2]);
                
                return {
                    prefix,
                    date,
                    sequence,
                    year: date.substring(0, 4),
                    month: date.substring(4, 6),
                    day: date.substring(6, 8),
                    full_number: number,
                    is_valid: true
                };
            }
            
            return {
                full_number: number,
                is_valid: false
            };
        }
        
        /**
         * Update UI with numbers
         */
        updateUINumbers(numbers) {
            // Update invoice displays
            document.querySelectorAll('[data-invoice-number]').forEach(el => {
                el.textContent = numbers.invoice;
                el.title = `Invoice: ${numbers.invoice}`;
            });
            
            // Update receipt displays
            document.querySelectorAll('[data-receipt-number]').forEach(el => {
                el.textContent = numbers.receipt;
                el.title = `Receipt: ${numbers.receipt}`;
            });
            
            // Update display numbers
            document.querySelectorAll('[data-display-number]').forEach(el => {
                el.textContent = numbers.invoice || numbers.receipt;
                el.title = `Invoice: ${numbers.invoice} | Receipt: ${numbers.receipt}`;
            });
            
            // Update payment modal
            const paymentInvoice = document.getElementById('paymentInvoiceNo');
            if (paymentInvoice) {
                paymentInvoice.textContent = numbers.invoice;
            }
            
            // Update confirmation
            const confirmInvoice = document.getElementById('confirmInvoice');
            if (confirmInvoice) {
                confirmInvoice.textContent = numbers.invoice;
            }
            
            // Log update
            console.log('📝 UI numbers updated:', numbers);
        }
        
        /**
         * Get number format information
         */
        getNumberFormat(type = 'invoice') {
            const formats = {
                invoice: {
                    prefix: 'INV',
                    pattern: 'INV-YYYYMMDD-NNN',
                    description: 'Invoice number for accounting',
                    example: 'INV-20231201-001'
                },
                receipt: {
                    prefix: 'RCPT',
                    pattern: 'RCPT-YYYYMMDD-NNN',
                    description: 'Receipt number for customer',
                    example: 'RCPT-20231201-001'
                },
                refund: {
                    prefix: 'REF',
                    pattern: 'REF-YYYYMMDD-NNN',
                    description: 'Refund transaction',
                    example: 'REF-20231201-001'
                },
                credit_note: {
                    prefix: 'CN',
                    pattern: 'CN-YYYYMMDD-NNN',
                    description: 'Credit note for returns',
                    example: 'CN-20231201-001'
                }
            };
            
            return formats[type] || formats.invoice;
        }
        
        /**
         * Generate a receipt preview with numbers
         */
        generateReceiptPreview(saleData) {
            const numbers = saleData.numbers || this.generateFallbackNumbers();
            const date = new Date(saleData.date || Date.now());
            
            return {
                header: {
                    store_name: saleData.store?.name || 'Kenyan Supermarket',
                    address: saleData.store?.address || 'Nairobi CBD',
                    phone: saleData.store?.phone || '0700 000 000',
                    date: date.toLocaleDateString('en-KE'),
                    time: date.toLocaleTimeString('en-KE', { hour: '2-digit', minute: '2-digit' })
                },
                numbers: {
                    invoice: numbers.invoice,
                    receipt: numbers.receipt,
                    display: numbers.invoice || numbers.receipt
                },
                customer: saleData.customer || {
                    name: 'Walk-in Customer',
                    phone: '',
                    vat_number: ''
                },
                items: saleData.items || [],
                totals: saleData.totals || {
                    subtotal: 0,
                    discount: 0,
                    vat_amount: 0,
                    grand_total: 0
                },
                payment: saleData.payment || {
                    method: 'cash',
                    amount_paid: 0,
                    change: 0
                },
                cashier: saleData.cashier || 'Cashier',
                footer: 'Thank you for your business!'
            };
        }
        
        // ==================== DOM & UI METHODS ====================
        
        createToastContainer() {
            if (this.toastContainer && document.body.contains(this.toastContainer)) {
                return this.toastContainer;
            }
            
            // Remove existing if broken
            const existing = document.getElementById('toastContainer');
            if (existing) {
                try {
                    existing.remove();
                } catch (e) {}
            }
            
            const container = document.createElement('div');
            container.id = 'toastContainer';
            container.className = 'position-fixed top-0 end-0 p-3';
            container.style.cssText = 'z-index: 9999; max-width: 350px; pointer-events: none;';
            
            document.body.appendChild(container);
            this.toastContainer = container;
            
            return container;
        }
        
        createLoadingOverlay() {
            let overlay = document.getElementById('loadingOverlay');
            if (!overlay) {
                overlay = document.createElement('div');
                overlay.id = 'loadingOverlay';
                overlay.className = 'loading-overlay';
                overlay.style.cssText = `
                    display: none;
                    position: fixed;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: rgba(0, 0, 0, 0.7);
                    z-index: 9998;
                    align-items: center;
                    justify-content: center;
                    backdrop-filter: blur(4px);
                `;
                
                const spinner = document.createElement('div');
                spinner.className = 'loading-spinner';
                spinner.style.cssText = `
                    width: 60px;
                    height: 60px;
                    border: 4px solid rgba(255, 255, 255, 0.3);
                    border-radius: 50%;
                    border-top-color: #2a4365;
                    animation: spin 1s linear infinite;
                `;
                
                const message = document.createElement('div');
                message.id = 'loadingMessage';
                message.className = 'loading-message';
                message.style.cssText = `
                    margin-top: 20px;
                    color: white;
                    font-size: 14px;
                    text-align: center;
                `;
                
                // Add animation keyframes
                if (!document.querySelector('#spin-animation')) {
                    const style = document.createElement('style');
                    style.id = 'spin-animation';
                    style.textContent = `
                        @keyframes spin {
                            to { transform: rotate(360deg); }
                        }
                    `;
                    document.head.appendChild(style);
                }
                
                overlay.appendChild(spinner);
                overlay.appendChild(message);
                document.body.appendChild(overlay);
            }
            
            return overlay;
        }
        
        showLoading(show = true, message = '') {
            try {
                const overlay = document.getElementById('loadingOverlay');
                const messageEl = document.getElementById('loadingMessage');
                
                if (overlay) {
                    overlay.style.display = show ? 'flex' : 'none';
                }
                
                if (messageEl && message) {
                    messageEl.textContent = message;
                }
                
                // Toggle body scrolling
                document.body.style.overflow = show ? 'hidden' : '';
                
                return this;
            } catch (error) {
                console.error('showLoading error:', error);
                return this;
            }
        }
        
        showToast(message, type = 'info', duration = 3000) {
            try {
                if (!this.toastContainer || !document.body.contains(this.toastContainer)) {
                    this.createToastContainer();
                }
                
                if (!this.toastContainer) {
                    console.warn('Toast container not available');
                    return null;
                }
                
                const toastId = 'toast-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
                const iconMap = {
                    success: 'bx-check-circle',
                    error: 'bx-error-circle',
                    warning: 'bx-error',
                    info: 'bx-info-circle'
                };
                
                const typeClass = type === 'error' ? 'danger' : type;
                const icon = iconMap[type] || 'bx-info-circle';
                
                const toastHtml = `
                    <div id="${toastId}" class="toast show" role="alert" 
                         style="pointer-events: auto; margin-bottom: 10px; animation: fadeIn 0.3s;">
                        <div class="toast-header bg-${typeClass} text-white border-0">
                            <i class="bx ${icon} me-2"></i>
                            <strong class="me-auto">${type.charAt(0).toUpperCase() + type.slice(1)}</strong>
                            <button type="button" class="btn-close btn-close-white" 
                                    onclick="document.getElementById('${toastId}').remove()"></button>
                        </div>
                        <div class="toast-body">
                            ${this.escapeHtml(message)}
                        </div>
                    </div>
                `;
                
                this.toastContainer.insertAdjacentHTML('beforeend', toastHtml);
                
                if (duration > 0) {
                    setTimeout(() => {
                        const toast = document.getElementById(toastId);
                        if (toast) toast.remove();
                    }, duration);
                }
                
                return toastId;
            } catch (error) {
                console.error('showToast error:', error);
                return null;
            }
        }
        
        // ==================== UTILITY METHODS ====================
        
        debounce(func, wait, immediate = false) {
            let timeout;
            return function executedFunction(...args) {
                const context = this;
                const later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        }
        
        throttle(func, limit) {
            let inThrottle;
            return function() {
                const args = arguments;
                const context = this;
                if (!inThrottle) {
                    func.apply(context, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            };
        }
        
        formatCurrency(amount, currency = 'KES') {
            if (amount === null || amount === undefined || amount === '') {
                return `${currency} 0.00`;
            }
            
            const num = parseFloat(amount);
            if (isNaN(num)) {
                return `${currency} 0.00`;
            }
            
            return `${currency} ${num.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
        }
        
        formatPhone(phone) {
            if (!phone) return '';
            const cleaned = phone.toString().replace(/\D/g, '');
            
            if (cleaned.length === 9 && /^[17]\d{8}$/.test(cleaned)) {
                return `0${cleaned}`;
            } else if (cleaned.length === 10 && cleaned.startsWith('0')) {
                return cleaned;
            } else if (cleaned.length === 12 && cleaned.startsWith('254')) {
                return `0${cleaned.slice(3)}`;
            } else if (cleaned.length === 13 && cleaned.startsWith('+254')) {
                return `0${cleaned.slice(4)}`;
            }
            
            return phone;
        }
        
        escapeHtml(text) {
            if (typeof text !== 'string') return text;
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // ==================== ERROR HANDLING ====================
        
        setupGlobalErrorHandling() {
            // Capture promise rejections
            window.addEventListener('unhandledrejection', (event) => {
                this.handleError(event.reason || new Error('Unhandled Promise Rejection'));
            });
            
            // Capture console errors
            if (console && console.error) {
                const originalError = console.error;
                console.error = (...args) => {
                    originalError.apply(console, args);
                    if (args[0] instanceof Error) {
                        this.handleError(args[0]);
                    }
                };
            }
        }
        
        handleError(error, context = '') {
            const errorData = {
                message: error.message || 'Unknown error',
                stack: error.stack,
                context: context,
                timestamp: new Date().toISOString(),
                url: window.location.href
            };
            
            console.error('🚨 Error handled:', errorData);
            
            // Don't show toast for toast-related errors
            if (!error.message?.includes('toast') && !error.message?.includes('insertAdjacentHTML')) {
                setTimeout(() => {
                    this.showToast(
                        `Error: ${error.message || 'Something went wrong'}`,
                        'error',
                        5000
                    );
                }, 100);
            }
            
            // Log to server
            this.logErrorToServer(errorData).catch(() => {
                // Silently fail
            });
            
            return errorData;
        }
        
        async logErrorToServer(errorData) {
            try {
                await fetch('/api/log-error', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': this.csrfToken
                    },
                    body: JSON.stringify(errorData)
                });
            } catch (e) {
                // Ignore logging errors
            }
        }
        
        // ==================== API METHODS ====================
        
        async fetchJson(url, options = {}) {
            const defaultOptions = {
                headers: {
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken,
                    'X-Requested-With': 'XMLHttpRequest'
                },
                credentials: 'same-origin'
            };
            
            const finalOptions = { ...defaultOptions, ...options };
            
            try {
                const response = await fetch(url, finalOptions);
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                return await response.json();
            } catch (error) {
                console.error(`Fetch error for ${url}:`, error);
                throw error;
            }
        }
        
        // ==================== MODAL METHODS ====================
        
        showModal(modalId) {
            try {
                const modalElement = document.getElementById(modalId);
                if (!modalElement) {
                    throw new Error(`Modal #${modalId} not found`);
                }
                
                const modal = bootstrap.Modal.getOrCreateInstance(modalElement);
                modal.show();
                return modal;
            } catch (error) {
                console.error(`Error showing modal ${modalId}:`, error);
                return null;
            }
        }
        
        hideModal(modalId) {
            try {
                const modalElement = document.getElementById(modalId);
                if (!modalElement) return false;
                
                const modal = bootstrap.Modal.getInstance(modalElement);
                if (modal) {
                    modal.hide();
                    return true;
                }
            } catch (error) {
                console.error(`Error hiding modal ${modalId}:`, error);
            }
            
            return false;
        }
        
        // ==================== CALCULATION METHODS ====================
        
        calculateVAT(amount, vatRate = 16) {
            const amountNum = parseFloat(amount) || 0;
            const vatAmount = amountNum * (vatRate / 100);
            const total = amountNum + vatAmount;
            
            return {
                amount: amountNum,
                vatRate: vatRate,
                vatAmount: parseFloat(vatAmount.toFixed(2)),
                total: parseFloat(total.toFixed(2))
            };
        }
        
        calculateDiscount(amount, discountPercent) {
            const amountNum = parseFloat(amount) || 0;
            const discount = amountNum * (discountPercent / 100);
            const discountedAmount = amountNum - discount;
            
            return {
                original: amountNum,
                discountPercent: discountPercent,
                discount: parseFloat(discount.toFixed(2)),
                discounted: parseFloat(discountedAmount.toFixed(2))
            };
        }
        
        calculateChange(tendered, total) {
            const tenderedNum = parseFloat(tendered) || 0;
            const totalNum = parseFloat(total) || 0;
            const change = tenderedNum - totalNum;
            
            return {
                tendered: tenderedNum,
                total: totalNum,
                change: Math.max(0, parseFloat(change.toFixed(2))),
                shortfall: Math.max(0, parseFloat((-change).toFixed(2)))
            };
        }
        
        // ==================== STORAGE METHODS ====================
        
        setLocalStorage(key, value) {
            try {
                localStorage.setItem(key, JSON.stringify(value));
                return true;
            } catch (error) {
                console.error('LocalStorage set error:', error);
                return false;
            }
        }
        
        getLocalStorage(key, defaultValue = null) {
            try {
                const item = localStorage.getItem(key);
                if (item === null) return defaultValue;
                return JSON.parse(item);
            } catch (error) {
                console.error('LocalStorage get error:', error);
                return defaultValue;
            }
        }
        
        // ==================== VALIDATION METHODS ====================
        
        validateEmail(email) {
            if (!email) return false;
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }
        
        validatePhone(phone) {
            const cleaned = this.formatPhone(phone).replace(/\D/g, '');
            return cleaned.length === 10 && /^0[17]\d{8}$/.test(cleaned);
        }
        
        validateRequired(field, fieldName) {
            if (!field || field.toString().trim() === '') {
                this.showToast(`${fieldName} is required`, 'error');
                return false;
            }
            return true;
        }
        
        // ==================== HELPER METHODS ====================
        
        sleep(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }
        
        generateId(prefix = 'id') {
            const timestamp = Date.now().toString(36);
            const random = Math.random().toString(36).substr(2, 9);
            return `${prefix}_${timestamp}_${random}`;
        }
        
        deepClone(obj) {
            return JSON.parse(JSON.stringify(obj));
        }
        
        getElementSafe(id) {
            const element = document.getElementById(id);
            if (!element) {
                console.warn(`Element #${id} not found`);
            }
            return element;
        }
        
        // ==================== DATE/TIME METHODS ====================
        
        formatDate(date, format = 'en-KE') {
            const d = new Date(date);
            return d.toLocaleDateString(format, {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            });
        }
        
        formatTime(date, format = 'en-KE') {
            const d = new Date(date);
            return d.toLocaleTimeString(format, {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
        }
        
        getCurrentDateString() {
            const date = new Date();
            return date.toISOString().slice(0,10).replace(/-/g,'');
        }
    }
    
    // Number Manager class
    class NumberManager {
        constructor() {
            this.prefixes = {
                invoice: 'INV',
                receipt: 'RCPT',
                refund: 'REF',
                credit_note: 'CN'
            };
        }
        
        generateNumber(prefix, sequence) {
            const date = new Date();
            const dateStr = date.toISOString().slice(0,10).replace(/-/g,'');
            const seqStr = sequence.toString().padStart(3, '0');
            return `${prefix}-${dateStr}-${seqStr}`;
        }
        
        parseNumber(number) {
            const parts = number.split('-');
            return {
                prefix: parts[0],
                date: parts[1],
                sequence: parseInt(parts[2]),
                year: parts[1]?.substring(0, 4),
                month: parts[1]?.substring(4, 6),
                day: parts[1]?.substring(6, 8)
            };
        }
        
        isValidNumber(number, type) {
            const prefix = this.prefixes[type];
            if (!prefix) return false;
            
            const regex = new RegExp(`^${prefix}-\\d{8}-\\d{3}$`);
            return regex.test(number);
        }
    }
    
    // Create instance
    const instance = new POSUtils();
    
    // Export all public methods
    const publicMethods = [
        'init', 'showLoading', 'showToast', 'debounce', 'throttle',
        'formatCurrency', 'formatPhone', 'escapeHtml', 'handleError',
        'fetchJson', 'showModal', 'hideModal', 'calculateVAT',
        'calculateDiscount', 'calculateChange', 'setLocalStorage',
        'getLocalStorage', 'validateEmail', 'validatePhone',
        'validateRequired', 'sleep', 'generateId', 'deepClone',
        'getElementSafe', 'formatDate', 'formatTime', 'getCurrentDateString',
        // Number management methods
        'generateNumbers', 'generateFallbackNumbers', 'validateNumber',
        'parseNumber', 'updateUINumbers', 'getNumberFormat',
        'generateReceiptPreview'
    ];
    
    // Create POSUtils object with all methods bound to instance
    window.POSUtils = {};
    
    publicMethods.forEach(methodName => {
        if (typeof instance[methodName] === 'function') {
            window.POSUtils[methodName] = function(...args) {
                return instance[methodName].apply(instance, args);
            };
        }
    });
    
    // Add version info
    window.POSUtils.version = instance.version;
    
    // Export NumberManager
    window.POSNumberManager = instance.numberManager;
    
    // For debugging
    window._POSUtilsInstance = instance;
    
    console.log('✅ POS Utils module loaded (v' + instance.version + ') with number management');
    
})();