// POS Products Management Module
// =========================================================

class POSProducts {
    static init() {
        this.setupSearchListeners();
        this.loadCategories();
        this.loadAll();
    }
    
    static setupSearchListeners() {
        const searchInput = document.getElementById('productSearch');
        if (!searchInput) return;
        
        let searchTimeout;
        
        searchInput.addEventListener('input', (e) => {
            clearTimeout(searchTimeout);
            const query = e.target.value.trim();
            
            if (query.length >= 2) {
                searchTimeout = setTimeout(() => {
                    this.searchWithPreview(query);
                }, 300);
            } else if (query.length === 0) {
                this.loadAll();
            }
        });
        
        searchInput.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                searchInput.value = '';
                this.loadAll();
            } else if (e.key === 'Enter') {
                this.searchWithPreview(searchInput.value);
            }
        });
        
        // Focus on page load
        setTimeout(() => {
            searchInput.focus();
            searchInput.select();
        }, 500);
    }
    
    static async loadAll(categoryId = null) {
        // SAFE loading - check if POSUtils exists
        if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
            window.POSUtils.showLoading(true, 'productGrid');
        } else {
            // Fallback loading
            const loadingOverlay = document.getElementById('loadingOverlay');
            if (loadingOverlay) {
                loadingOverlay.style.display = 'flex';
            }
        }
        
        try {
            let url = '/api/products';
            if (categoryId && categoryId !== 'all') {
                url += `?category_id=${categoryId}`;
            }
            
            const response = await fetch(url);
            const data = await response.json();
            
            // SAFE unloading
            if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
                window.POSUtils.showLoading(false, 'productGrid');
            } else {
                const loadingOverlay = document.getElementById('loadingOverlay');
                if (loadingOverlay) {
                    loadingOverlay.style.display = 'none';
                }
            }
            
            if (data.success) {
                this.render(data.products);
            } else {
                this.safeToast('Failed to load products', 'error');
            }
        } catch (error) {
            // SAFE unloading on error
            if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
                window.POSUtils.showLoading(false, 'productGrid');
            } else {
                const loadingOverlay = document.getElementById('loadingOverlay');
                if (loadingOverlay) {
                    loadingOverlay.style.display = 'none';
                }
            }
            console.error('Error loading products:', error);
            this.safeToast('Failed to load products', 'error');
        }
    }
    
    static async search(query) {
        if (!query || query.trim() === '') {
            this.loadAll();
            return;
        }
        
        try {
            const response = await fetch(`/api/products/search?q=${encodeURIComponent(query)}`);
            const data = await response.json();
            
            if (data.success) {
                this.render(data.products);
            } else {
                this.safeToast('Search failed', 'error');
            }
        } catch (error) {
            console.error('Search error:', error);
            this.safeToast('Search failed', 'error');
        }
    }
    
    static async searchWithPreview(query) {
        if (!query || query.trim() === '') {
            this.loadAll();
            return;
        }
        
        // SAFE loading
        if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
            window.POSUtils.showLoading(true);
        }
        
        try {
            const response = await fetch(`/api/products/search?q=${encodeURIComponent(query)}&limit=50`);
            const data = await response.json();
            
            // SAFE unloading
            if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
                window.POSUtils.showLoading(false);
            }
            
            if (data.success) {
                this.render(data.products);
                this.updateSearchStats(data.count);
            } else {
                this.safeToast('Search failed', 'error');
            }
        } catch (error) {
            // SAFE unloading on error
            if (window.POSUtils && typeof window.POSUtils.showLoading === 'function') {
                window.POSUtils.showLoading(false);
            }
            console.error('Search error:', error);
            this.safeToast('Search failed', 'error');
        }
    }
    
    static render(products) {
        const container = document.getElementById('productGrid');
        if (!container) return;
        
        if (!products || products.length === 0) {
            container.innerHTML = `
                <div class="col-12 text-center py-5">
                    <i class="bx bx-package bx-lg text-muted mb-3"></i>
                    <p class="text-muted">No products found</p>
                    <button class="btn btn-sm btn-outline-primary" onclick="this.safeToast('Add product feature coming soon', 'info')">
                        Add New Product
                    </button>
                </div>
            `;
            return;
        }
        
        container.innerHTML = '';
        
        products.forEach(product => {
            const productCard = this.createProductCard(product);
            container.appendChild(productCard);
        });
    }
    
    static createProductCard(product) {
        const template = document.getElementById('productCardTemplate');
        if (!template) {
            const div = document.createElement('div');
            div.innerHTML = '<div class="text-danger">Product card template not found</div>';
            return div;
        }
        
        const card = template.content.cloneNode(true);
        const cardElement = card.querySelector('.product-card');
        
        // Store product data
        cardElement._productData = product;
        
        // Fill product data
        card.querySelector('[data-field="name"]').textContent = product.name || 'Unknown Product';
        card.querySelector('[data-field="sku"]').textContent = product.sku || 'N/A';
        card.querySelector('[data-field="barcode"]').textContent = product.barcode || 'N/A';
        
        // Format price safely
        const price = product.price || product.final_price || product.sale_price || 0;
        const cost = product.cost_price || 0;
        card.querySelector('[data-field="price"]').textContent = this.formatCurrency(price);
        card.querySelector('[data-field="cost"]').textContent = `Cost: ${this.formatCurrency(cost)}`;
        card.querySelector('[data-field="stock"]').textContent = `${product.stock || 0} in stock`;
        
        // Update badges
        const stockBadge = card.querySelector('[data-badge-type="stock"]');
        const vatBadge = card.querySelector('[data-badge-type="vat"]');
        const lowStockBadge = card.querySelector('[data-badge-type="low-stock"]');
        
        if (!product.stock || product.stock <= 0) {
            stockBadge.textContent = 'Out of Stock';
            stockBadge.className = 'badge bg-danger';
            cardElement.classList.add('out-of-stock');
        } else if (product.stock <= 5) {
            stockBadge.textContent = 'Low Stock';
            stockBadge.className = 'badge bg-warning';
        } else {
            stockBadge.textContent = 'In Stock';
            stockBadge.className = 'badge bg-success';
        }
        
        if (product.has_vat || product.is_vatable) {
            vatBadge.classList.remove('d-none');
        }
        
        if (product.stock <= (product.low_stock_threshold || 5)) {
            lowStockBadge.classList.remove('d-none');
        }
        
        // Event listeners
        cardElement.addEventListener('click', (e) => {
            if (cardElement.classList.contains('out-of-stock')) return;
            if (e.target.closest('.btn') || e.target.closest('.btn-group')) return;
            
            this.addToCart(product, 1);
        });
        
        cardElement.addEventListener('keypress', (e) => {
            if ((e.key === 'Enter' || e.key === ' ') && !cardElement.classList.contains('out-of-stock')) {
                this.addToCart(product, 1);
            }
        });
        
        const addBtn = cardElement.querySelector('.btn-outline-primary');
        if (addBtn) {
            addBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                if (!cardElement.classList.contains('out-of-stock')) {
                    this.addToCart(product, 1);
                }
            });
        }
        
        const detailsBtn = cardElement.querySelector('[onclick*="viewDetails"]');
        if (detailsBtn) {
            detailsBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.viewDetails(product.id);
            });
        }
        
        const favoritesBtn = cardElement.querySelector('[onclick*="addToFavorites"]');
        if (favoritesBtn) {
            favoritesBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.addToFavorites(product.id);
            });
        }
        
        return card;
    }
    
    static addToCart(product, quantity = 1) {
        if (window.POSCart && typeof window.POSCart.add === 'function') {
            window.POSCart.add(product, quantity);
        } else if (window.posState) {
            // Fallback: directly add to posState
            if (!window.posState.cart) window.posState.cart = [];
            const existingIndex = window.posState.cart.findIndex(item => item.id === product.id);
            
            if (existingIndex > -1) {
                window.posState.cart[existingIndex].quantity += quantity;
            } else {
                window.posState.cart.push({
                    ...product,
                    quantity: quantity
                });
            }
            
            // Update cart display if available
            if (window.POSCart && window.POSCart.updateDisplay) {
                window.POSCart.updateDisplay();
            }
            
            this.safeToast(`${product.name} added to cart`, 'success');
        }
    }
    
    static updateSearchStats(count) {
        let searchStats = document.getElementById('searchStats');
        if (!searchStats) {
            searchStats = document.createElement('div');
            searchStats.id = 'searchStats';
            searchStats.className = 'position-fixed top-0 start-50 translate-middle-x mt-3';
            searchStats.style.zIndex = '999';
            document.body.appendChild(searchStats);
        }
        
        searchStats.innerHTML = `
            <div class="badge bg-info">
                <i class="bx bx-search me-1"></i> ${count} products found
            </div>
        `;
        
        setTimeout(() => {
            searchStats.innerHTML = '';
        }, 3000);
    }
    
 static async loadCategories() {
    try {
        const response = await fetch('/api/categories');
        
        // Check if response is JSON
        const contentType = response.headers.get('content-type');
        if (!response.ok || !contentType || !contentType.includes('application/json')) {
            throw new Error('Failed to load categories');
        }
        
        const data = await response.json();
        
        if (data.success) {
            this.renderCategories(data.categories);
        }
    } catch (error) {
        console.error('Error loading categories:', error);
        // Don't show error to user, just use empty categories
        this.renderCategories([]);
    }
}
    
    static renderCategories(categories) {
        const filter = document.getElementById('categoryFilter');
        if (!filter) return;
        
        let html = '<option value="all">All Categories</option>';
        categories.forEach(category => {
            html += `<option value="${category.id}">${category.name}</option>`;
        });
        
        filter.innerHTML = html;
    }
    
    static filterByCategory() {
        const categoryFilter = document.getElementById('categoryFilter');
        if (categoryFilter) {
            this.loadAll(categoryFilter.value);
        }
    }
    
    static viewDetails(productId) {
        console.log('View product details:', productId);
        this.safeToast('Product details feature coming soon!', 'info');
    }
    
    static addToFavorites(productId) {
        console.log('Add to favorites:', productId);
        this.safeToast('Added to favorites', 'success');
    }
    
    static showLowStock() {
        this.loadAll();
        this.safeToast('Showing low stock items', 'info');
    }
    
    // Safe helper methods
static safeToast(message, type = 'info') {
    // Try window.POSUtils first
    if (window.POSUtils && typeof window.POSUtils.showToast === 'function') {
        try {
            return window.POSUtils.showToast(message, type);
        } catch (e) {
            console.error('POSUtils.showToast failed:', e);
        }
    }
    
    // Try window.safeToast fallback
    if (window.safeToast && typeof window.safeToast === 'function') {
        try {
            return window.safeToast(message, type);
        } catch (e) {
            console.error('window.safeToast failed:', e);
        }
    }
    
    // Last resort: console log
    console.log(`💬 ${type.toUpperCase()}: ${message}`);
    
    // Try to create a simple toast
    try {
        const container = document.getElementById('toastContainer');
        if (!container) {
            const newContainer = document.createElement('div');
            newContainer.id = 'toastContainer';
            newContainer.className = 'position-fixed top-0 end-0 p-3';
            newContainer.style.cssText = 'z-index: 9999; max-width: 350px; pointer-events: none;';
            document.body.appendChild(newContainer);
        }
        
        const toastId = 'simple-toast-' + Date.now();
        const toastHtml = `
            <div id="${toastId}" class="toast show" style="pointer-events: auto; margin-bottom: 10px;">
                <div class="toast-body">
                    <i class="bx bx-${type === 'success' ? 'check-circle' : type === 'error' ? 'error-circle' : type === 'warning' ? 'error' : 'info-circle'} me-2"></i>
                    ${message}
                </div>
            </div>
        `;
        
        document.getElementById('toastContainer').insertAdjacentHTML('beforeend', toastHtml);
        
        setTimeout(() => {
            const toast = document.getElementById(toastId);
            if (toast) toast.remove();
        }, 3000);
        
    } catch (e) {
        console.error('Failed to create fallback toast:', e);
    }
}
    
    static formatCurrency(amount) {
        if (window.POSUtils && typeof window.POSUtils.formatCurrency === 'function') {
            return window.POSUtils.formatCurrency(amount);
        }
        return `KES ${parseFloat(amount || 0).toFixed(2)}`;
    }
}

// Export for global access
window.POSProducts = POSProducts;
window.searchProductsWithPreview = (query) => POSProducts.searchWithPreview(query);
window.filterByCategory = () => POSProducts.filterByCategory();
window.showLowStock = () => POSProducts.showLowStock();