// =========================================================
// M-Pesa Payment Processor
// =========================================================

import BasePaymentProcessor from './base-processor.js';

class MpesaProcessor extends BasePaymentProcessor {
    constructor() {
        super('mpesa', 'M-Pesa');
        this.setConfig({
            stkPushEnabled: true,
            manualEntryEnabled: true,
            autoConfirm: true,
            sendSMS: true,
            maxAmount: 150000,
            minAmount: 10,
            callbackUrl: '/api/mpesa/callback',
            pollInterval: 2000,
            maxPollAttempts: 30
        });
        
        this.pollingIntervals = new Map();
    }

    // ========== VALIDATION ==========
    async validate(data) {
        try {
            const requiredFields = ['amount', 'phone'];
            if (data.type === 'manual') {
                requiredFields.push('transactionId');
            }
            
            this.validateRequiredFields(data, requiredFields);
            
            const amount = this.validateAmount(data.amount);
            const phone = this.validatePhone(data.phone);
            
            // Check amount limits
            if (amount < this.getConfig('minAmount')) {
                throw this.createError(
                    'AMOUNT_TOO_SMALL',
                    `Amount (${this.formatCurrency(amount)}) is below minimum (${this.formatCurrency(this.getConfig('minAmount'))})`
                );
            }
            
            if (amount > this.getConfig('maxAmount')) {
                throw this.createError(
                    'AMOUNT_TOO_LARGE',
                    `Amount (${this.formatCurrency(amount)}) exceeds maximum (${this.formatCurrency(this.getConfig('maxAmount'))})`
                );
            }
            
            return this.formatResponse(true, {
                amount: amount,
                phone: phone,
                type: data.type || 'stk',
                validated: true
            }, 'M-Pesa validation successful');
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== PROCESSING ==========
    async process(data) {
        try {
            // Validate
            const validation = await this.validate(data);
            if (!validation.success) {
                throw new Error(validation.message);
            }
            
            const amount = parseFloat(data.amount);
            const phone = this.formatPhone(data.phone);
            const type = data.type || 'stk';
            
            // Log transaction start
            this.logTransaction('start', {
                amount: amount,
                phone: phone,
                type: type,
                customer: data.customer
            });
            
            let result;
            
            if (type === 'stk') {
                result = await this.processSTKPush(amount, phone, data);
            } else {
                result = await this.processManualEntry(data);
            }
            
            // Create transaction record
            const transaction = {
                id: result.transactionId || `MPESA_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
                method: this.methodCode,
                type: type,
                amount: amount,
                phone: phone,
                timestamp: new Date().toISOString(),
                status: 'pending',
                customer: data.customer || null,
                items: data.items || [],
                invoiceNo: data.invoiceNo || this.generateInvoiceNumber(),
                ...result
            };
            
            // Save transaction
            await this.saveTransaction(transaction);
            
            // Poll for confirmation if STK
            if (type === 'stk' && this.getConfig('autoConfirm')) {
                transaction.pollingId = await this.startPolling(transaction.id, result.checkoutRequestId);
            }
            
            this.logTransaction(type === 'stk' ? 'stk_initiated' : 'manual_entered', transaction);
            
            return this.formatResponse(true, {
                transaction: transaction,
                nextStep: type === 'stk' ? 'polling' : 'completed'
            }, type === 'stk' 
                ? 'STK Push initiated. Waiting for customer confirmation...'
                : 'Manual M-Pesa entry recorded');
            
        } catch (error) {
            this.handleError(error, 'mpesa processing');
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== STK PUSH PROCESSING ==========
    async processSTKPush(amount, phone, data) {
        try {
            const stkData = {
                phone: phone,
                amount: amount,
                accountReference: data.reference || `POS_${Date.now()}`,
                transactionDesc: data.description || 'POS Payment',
                callbackUrl: this.getConfig('callbackUrl')
            };
            
            const response = await this.callAPI('/api/mpesa/stk-push', stkData);
            
            if (!response.success) {
                throw this.createError(
                    'STK_PUSH_FAILED',
                    response.message || 'Failed to initiate STK Push'
                );
            }
            
            return {
                checkoutRequestId: response.CheckoutRequestID,
                merchantRequestId: response.MerchantRequestID,
                responseCode: response.ResponseCode,
                responseDescription: response.ResponseDescription,
                customerMessage: response.CustomerMessage,
                transactionId: `MPESA_STK_${Date.now()}`
            };
            
        } catch (error) {
            this.handleError(error, 'stk push');
            throw error;
        }
    }

    // ========== MANUAL ENTRY PROCESSING ==========
    async processManualEntry(data) {
        try {
            // Verify the transaction exists
            const verification = await this.verifyTransaction(data.transactionId, data.phone);
            
            if (!verification.success) {
                throw this.createError(
                    'TRANSACTION_NOT_FOUND',
                    verification.message || 'M-Pesa transaction not found'
                );
            }
            
            return {
                transactionId: data.transactionId,
                receiptNo: data.receiptNo || verification.receiptNo,
                confirmed: true,
                verificationData: verification.data
            };
            
        } catch (error) {
            this.handleError(error, 'manual entry');
            throw error;
        }
    }

    // ========== POLLING ==========
    async startPolling(transactionId, checkoutRequestId) {
        const pollingId = `poll_${transactionId}`;
        
        let attempts = 0;
        const maxAttempts = this.getConfig('maxPollAttempts');
        const interval = this.getConfig('pollInterval');
        
        const poll = async () => {
            attempts++;
            
            if (attempts > maxAttempts) {
                this.stopPolling(pollingId);
                await this.updateTransactionStatus(transactionId, 'timeout');
                this.trigger('pollingTimeout', { transactionId, attempts });
                return;
            }
            
            try {
                const status = await this.checkSTKStatus(checkoutRequestId);
                
                if (status.success) {
                    this.stopPolling(pollingId);
                    
                    // Update transaction with full details
                    const transaction = await this.getTransaction(transactionId);
                    if (transaction) {
                        transaction.status = 'completed';
                        transaction.completedAt = new Date().toISOString();
                        transaction.mpesaDetails = status.data;
                        transaction.receiptNo = status.receiptNo;
                        
                        await this.updateTransaction(transactionId, transaction);
                        
                        // Send SMS if enabled
                        if (this.getConfig('sendSMS')) {
                            await this.sendConfirmationSMS(transaction);
                        }
                        
                        this.trigger('paymentConfirmed', { transaction });
                        this.logTransaction('confirmed', transaction);
                    }
                    
                } else if (status.code === '1037') {
                    // Request cancelled by user - stop polling
                    this.stopPolling(pollingId);
                    await this.updateTransactionStatus(transactionId, 'cancelled');
                    this.trigger('paymentCancelled', { transactionId });
                    
                } else if (status.code !== '1032') {
                    // 1032 means "Request processing" - continue polling
                    // Other error codes might indicate issues
                    console.warn(`STK status error: ${status.code} - ${status.description}`);
                }
                
            } catch (error) {
                console.error('Polling error:', error);
            }
        };
        
        // Start polling
        const intervalId = setInterval(poll, interval);
        this.pollingIntervals.set(pollingId, intervalId);
        
        // Initial poll
        setTimeout(poll, 1000);
        
        return pollingId;
    }

    stopPolling(pollingId) {
        if (this.pollingIntervals.has(pollingId)) {
            clearInterval(this.pollingIntervals.get(pollingId));
            this.pollingIntervals.delete(pollingId);
        }
    }

    // ========== VERIFICATION ==========
    async verify(transactionId) {
        try {
            const transaction = await this.getTransaction(transactionId);
            
            if (!transaction) {
                throw this.createError(
                    'TRANSACTION_NOT_FOUND',
                    `Transaction ${transactionId} not found`
                );
            }
            
            if (transaction.type === 'stk') {
                return await this.verifySTKTransaction(transaction);
            } else {
                return await this.verifyManualTransaction(transaction);
            }
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    async verifySTKTransaction(transaction) {
        try {
            if (!transaction.checkoutRequestId) {
                throw this.createError(
                    'MISSING_CHECKOUT_ID',
                    'No checkout request ID found for STK transaction'
                );
            }
            
            const status = await this.checkSTKStatus(transaction.checkoutRequestId);
            
            return this.formatResponse(true, {
                transaction: transaction,
                status: status,
                verified: status.success,
                receiptNo: status.receiptNo
            }, 'STK transaction verified');
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    async verifyManualTransaction(transaction) {
        try {
            const verification = await this.verifyTransaction(
                transaction.transactionId,
                transaction.phone
            );
            
            return this.formatResponse(true, {
                transaction: transaction,
                verification: verification,
                verified: verification.success
            }, 'Manual transaction verified');
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    async verifyTransaction(transactionId, phone) {
        try {
            const response = await this.callAPI('/api/mpesa/verify', {
                transactionId: transactionId,
                phone: phone
            });
            
            return response;
            
        } catch (error) {
            throw this.createError(
                'VERIFICATION_FAILED',
                'Failed to verify M-Pesa transaction',
                { transactionId, phone, error: error.message }
            );
        }
    }

    // ========== STATUS CHECKING ==========
    async checkSTKStatus(checkoutRequestId) {
        try {
            const response = await this.callAPI('/api/mpesa/stk-status', {
                checkoutRequestId: checkoutRequestId
            });
            
            return response;
            
        } catch (error) {
            throw this.createError(
                'STATUS_CHECK_FAILED',
                'Failed to check STK status',
                { checkoutRequestId, error: error.message }
            );
        }
    }

    // ========== REFUND ==========
    async refund(transactionId, amount, reason = '') {
        try {
            const transaction = await this.getTransaction(transactionId);
            
            if (!transaction) {
                throw this.createError(
                    'TRANSACTION_NOT_FOUND',
                    `Transaction ${transactionId} not found`
                );
            }
            
            if (transaction.status !== 'completed') {
                throw this.createError(
                    'INVALID_STATUS',
                    `Transaction ${transactionId} cannot be refunded (status: ${transaction.status})`
                );
            }
            
            const refundAmount = this.validateAmount(amount);
            
            if (refundAmount > transaction.amount) {
                throw this.createError(
                    'EXCESSIVE_REFUND',
                    `Refund amount (${this.formatCurrency(refundAmount)}) exceeds transaction amount (${this.formatCurrency(transaction.amount)})`
                );
            }
            
            // Initiate M-Pesa reversal
            const reversal = await this.initiateReversal(transaction, refundAmount, reason);
            
            // Create refund record
            const refund = {
                id: `REFUND_${transactionId}_${Date.now()}`,
                originalTransaction: transactionId,
                method: this.methodCode,
                amount: refundAmount,
                reason: reason,
                timestamp: new Date().toISOString(),
                status: 'pending',
                reversalId: reversal.conversationId,
                processedBy: data.userId || 'system'
            };
            
            // Save refund
            await this.saveRefund(refund);
            
            // Update original transaction
            transaction.refunds = transaction.refunds || [];
            transaction.refunds.push(refund);
            
            await this.updateTransaction(transactionId, transaction);
            
            this.logTransaction('refund_initiated', { refund, reversal });
            
            // Start polling for reversal status
            this.startRefundPolling(refund.id, reversal.conversationId);
            
            return this.formatResponse(true, {
                refund: refund,
                reversal: reversal,
                transaction: transaction
            }, `M-Pesa refund initiated for ${this.formatCurrency(refundAmount)}`);
            
        } catch (error) {
            this.handleError(error, 'mpesa refund');
            return this.formatResponse(false, null, error.message);
        }
    }

    async initiateReversal(transaction, amount, reason) {
        try {
            const reversalData = {
                transactionId: transaction.transactionId,
                amount: amount,
                reason: reason,
                remarks: `Refund for transaction ${transaction.id}: ${reason}`
            };
            
            const response = await this.callAPI('/api/mpesa/reversal', reversalData);
            
            if (!response.success) {
                throw this.createError(
                    'REVERSAL_FAILED',
                    response.message || 'Failed to initiate M-Pesa reversal'
                );
            }
            
            return {
                conversationId: response.ConversationID,
                originatorConversationId: response.OriginatorConversationID,
                responseCode: response.ResponseCode,
                responseDescription: response.ResponseDescription
            };
            
        } catch (error) {
            this.handleError(error, 'reversal initiation');
            throw error;
        }
    }

    async startRefundPolling(refundId, conversationId) {
        const pollingId = `refund_poll_${refundId}`;
        let attempts = 0;
        const maxAttempts = 30;
        
        const poll = async () => {
            attempts++;
            
            if (attempts > maxAttempts) {
                this.stopPolling(pollingId);
                await this.updateRefundStatus(refundId, 'timeout');
                return;
            }
            
            try {
                const status = await this.checkReversalStatus(conversationId);
                
                if (status.success) {
                    this.stopPolling(pollingId);
                    await this.updateRefundStatus(refundId, 'completed', status);
                    this.trigger('refundCompleted', { refundId, status });
                }
                
            } catch (error) {
                console.error('Refund polling error:', error);
            }
        };
        
        const intervalId = setInterval(poll, 5000);
        this.pollingIntervals.set(pollingId, intervalId);
        
        setTimeout(poll, 2000);
        
        return pollingId;
    }

    async checkReversalStatus(conversationId) {
        try {
            const response = await this.callAPI('/api/mpesa/reversal-status', {
                conversationId: conversationId
            });
            
            return response;
            
        } catch (error) {
            throw this.createError(
                'REVERSAL_STATUS_FAILED',
                'Failed to check reversal status',
                { conversationId, error: error.message }
            );
        }
    }

    // ========== SMS CONFIRMATION ==========
    async sendConfirmationSMS(transaction) {
        try {
            if (!transaction.phone) return;
            
            const message = this.generateConfirmationSMS(transaction);
            
            const response = await this.callAPI('/api/sms/send', {
                phone: transaction.phone,
                message: message,
                transactionId: transaction.id
            });
            
            if (response.success) {
                this.logTransaction('sms_sent', {
                    transactionId: transaction.id,
                    phone: transaction.phone,
                    messageId: response.messageId
                });
            }
            
            return response;
            
        } catch (error) {
            console.warn('Failed to send SMS:', error.message);
            return { success: false, error: error.message };
        }
    }

    generateConfirmationSMS(transaction) {
        const amount = this.formatCurrency(transaction.amount);
        const date = new Date(transaction.timestamp).toLocaleString('en-KE', {
            dateStyle: 'medium',
            timeStyle: 'short'
        });
        const receipt = transaction.receiptNo || transaction.id.slice(-8);
        
        return `Confirmed. Ksh${amount} sent to ${transaction.customer?.business || 'Merchant'} on ${date}. Receipt: ${receipt}. Thank you!`;
    }

    // ========== HELPER METHODS ==========
    generateInvoiceNumber() {
        const date = new Date();
        const year = date.getFullYear().toString().slice(-2);
        const month = (date.getMonth() + 1).toString().padStart(2, '0');
        const day = date.getDate().toString().padStart(2, '0');
        const random = Math.floor(Math.random() * 10000).toString().padStart(4, '0');
        return `MPESA${year}${month}${day}${random}`;
    }

    async saveTransaction(transaction) {
        try {
            // Save locally
            const transactions = JSON.parse(localStorage.getItem('mpesa_transactions') || '[]');
            transactions.push(transaction);
            localStorage.setItem('mpesa_transactions', JSON.stringify(transactions));
            
            // Send to server
            const result = await this.callAPI('/api/payments/mpesa/save', { transaction });
            
            if (result.success) {
                transaction.serverId = result.transactionId;
                this.updateLocalTransaction(transaction.id, transaction);
            }
            
            return result;
            
        } catch (error) {
            console.warn('Failed to save transaction to server:', error.message);
            return { success: true, offline: true };
        }
    }

    async getTransaction(transactionId) {
        // Try local storage first
        const transactions = JSON.parse(localStorage.getItem('mpesa_transactions') || '[]');
        let transaction = transactions.find(t => t.id === transactionId);
        
        // Try server if not found locally
        if (!transaction) {
            try {
                const result = await this.callAPI('/api/payments/mpesa/get', { transactionId });
                if (result.success && result.transaction) {
                    transaction = result.transaction;
                }
            } catch (error) {
                console.warn('Failed to fetch transaction from server:', error.message);
            }
        }
        
        return transaction;
    }

    async updateTransaction(transactionId, updates) {
        // Update locally
        const transactions = JSON.parse(localStorage.getItem('mpesa_transactions') || '[]');
        const index = transactions.findIndex(t => t.id === transactionId);
        
        if (index > -1) {
            transactions[index] = { ...transactions[index], ...updates };
            localStorage.setItem('mpesa_transactions', JSON.stringify(transactions));
        }
        
        // Update on server
        try {
            await this.callAPI('/api/payments/mpesa/update', {
                transactionId: transactionId,
                updates: updates
            });
        } catch (error) {
            console.warn('Failed to update transaction on server:', error.message);
        }
    }

    async updateTransactionStatus(transactionId, status, details = {}) {
        return this.updateTransaction(transactionId, {
            status: status,
            ...details,
            updatedAt: new Date().toISOString()
        });
    }

    async saveRefund(refund) {
        try {
            const refunds = JSON.parse(localStorage.getItem('mpesa_refunds') || '[]');
            refunds.push(refund);
            localStorage.setItem('mpesa_refunds', JSON.stringify(refunds));
            
            await this.callAPI('/api/payments/mpesa/refund/save', { refund });
            
        } catch (error) {
            console.warn('Failed to save refund to server:', error.message);
        }
    }

    async updateRefundStatus(refundId, status, details = {}) {
        const refunds = JSON.parse(localStorage.getItem('mpesa_refunds') || '[]');
        const index = refunds.findIndex(r => r.id === refundId);
        
        if (index > -1) {
            refunds[index] = { ...refunds[index], status: status, ...details };
            localStorage.setItem('mpesa_refunds', JSON.stringify(refunds));
        }
        
        try {
            await this.callAPI('/api/payments/mpesa/refund/update', {
                refundId: refundId,
                status: status,
                details: details
            });
        } catch (error) {
            console.warn('Failed to update refund on server:', error.message);
        }
    }

    updateLocalTransaction(transactionId, updates) {
        const transactions = JSON.parse(localStorage.getItem('mpesa_transactions') || '[]');
        const index = transactions.findIndex(t => t.id === transactionId);
        
        if (index > -1) {
            transactions[index] = { ...transactions[index], ...updates };
            localStorage.setItem('mpesa_transactions', JSON.stringify(transactions));
        }
    }

    // ========== M-PESA STATEMENTS ==========
    async getStatement(startDate, endDate) {
        try {
            const response = await this.callAPI('/api/mpesa/statement', {
                startDate: startDate,
                endDate: endDate
            });
            
            return response;
            
        } catch (error) {
            throw this.createError(
                'STATEMENT_FAILED',
                'Failed to fetch M-Pesa statement',
                { startDate, endDate, error: error.message }
            );
        }
    }

    // ========== CLEANUP ==========
    destroy() {
        // Clear all polling intervals
        this.pollingIntervals.forEach((intervalId, pollingId) => {
            clearInterval(intervalId);
        });
        this.pollingIntervals.clear();
        
        super.destroy();
    }
}

export default MpesaProcessor;