// =========================================================
// Cash Payment Processor
// =========================================================

import BasePaymentProcessor from './base-processor.js';

class CashProcessor extends BasePaymentProcessor {
    constructor() {
        super('cash', 'Cash');
        this.setConfig({
            allowPartialPayment: false,
            requireExactAmount: false,
            maxChangeAmount: 10000,
            denominations: [50, 100, 200, 500, 1000, 2000, 5000]
        });
    }

    // ========== VALIDATION ==========
    async validate(data) {
        try {
            this.validateRequiredFields(data, ['amount', 'received']);
            
            const amount = this.validateAmount(data.amount);
            const received = this.validateAmount(data.received);
            
            if (!this.getConfig('allowPartialPayment') && received < amount) {
                throw this.createError(
                    'INSUFFICIENT_CASH',
                    `Insufficient cash. Required: ${this.formatCurrency(amount)}, Received: ${this.formatCurrency(received)}`,
                    { amount, received, shortfall: amount - received }
                );
            }
            
            const change = received - amount;
            if (change > this.getConfig('maxChangeAmount')) {
                throw this.createError(
                    'EXCESSIVE_CHANGE',
                    `Change amount (${this.formatCurrency(change)}) exceeds maximum allowed (${this.formatCurrency(this.getConfig('maxChangeAmount'))})`,
                    { change, maxChange: this.getConfig('maxChangeAmount') }
                );
            }
            
            return this.formatResponse(true, {
                amount: amount,
                received: received,
                change: Math.max(change, 0),
                exact: Math.abs(change) < 0.01
            }, 'Cash validation successful');
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== PROCESSING ==========
    async process(data) {
        try {
            // Validate first
            const validation = await this.validate(data);
            if (!validation.success) {
                throw new Error(validation.message);
            }
            
            const amount = parseFloat(data.amount);
            const received = parseFloat(data.received || amount);
            const change = Math.max(received - amount, 0);
            
            // Log transaction start
            this.logTransaction('start', {
                amount: amount,
                received: received,
                change: change,
                customer: data.customer,
                items: data.items?.length || 0
            });
            
            // Create transaction record
            const transaction = {
                id: `CASH_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
                method: this.methodCode,
                amount: amount,
                received: received,
                change: change,
                timestamp: new Date().toISOString(),
                notes: data.notes || '',
                status: 'completed',
                customer: data.customer || null,
                items: data.items || [],
                invoiceNo: data.invoiceNo || this.generateInvoiceNumber()
            };
            
            // Save transaction
            await this.saveTransaction(transaction);
            
            // Calculate denominations for change
            if (change > 0) {
                transaction.changeDenominations = this.calculateChangeDenominations(change);
            }
            
            // Log completion
            this.logTransaction('complete', transaction);
            
            // Trigger events
            this.trigger('processed', { transaction });
            
            return this.formatResponse(true, {
                transaction: transaction,
                receipt: this.generateReceipt(transaction)
            }, 'Cash payment processed successfully');
            
        } catch (error) {
            this.handleError(error, 'cash processing');
            
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== VERIFICATION ==========
    async verify(transactionId) {
        try {
            const transaction = await this.getTransaction(transactionId);
            
            if (!transaction) {
                throw this.createError(
                    'TRANSACTION_NOT_FOUND',
                    `Transaction ${transactionId} not found`
                );
            }
            
            return this.formatResponse(true, {
                transaction: transaction,
                verified: true,
                status: transaction.status
            }, 'Transaction verified');
            
        } catch (error) {
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== REFUND ==========
    async refund(transactionId, amount, reason = '') {
        try {
            const transaction = await this.getTransaction(transactionId);
            
            if (!transaction) {
                throw this.createError(
                    'TRANSACTION_NOT_FOUND',
                    `Transaction ${transactionId} not found`
                );
            }
            
            if (transaction.status !== 'completed') {
                throw this.createError(
                    'INVALID_STATUS',
                    `Transaction ${transactionId} cannot be refunded (status: ${transaction.status})`
                );
            }
            
            const refundAmount = this.validateAmount(amount);
            
            if (refundAmount > transaction.amount) {
                throw this.createError(
                    'EXCESSIVE_REFUND',
                    `Refund amount (${this.formatCurrency(refundAmount)}) exceeds transaction amount (${this.formatCurrency(transaction.amount)})`
                );
            }
            
            // Create refund transaction
            const refund = {
                id: `REFUND_${transactionId}_${Date.now()}`,
                originalTransaction: transactionId,
                method: this.methodCode,
                amount: refundAmount,
                reason: reason,
                timestamp: new Date().toISOString(),
                status: 'completed',
                processedBy: data.userId || 'system'
            };
            
            // Save refund
            await this.saveRefund(refund);
            
            // Update original transaction
            transaction.refunds = transaction.refunds || [];
            transaction.refunds.push(refund);
            transaction.status = refundAmount === transaction.amount ? 'refunded' : 'partially_refunded';
            
            await this.updateTransaction(transactionId, transaction);
            
            this.logTransaction('refund', refund);
            
            return this.formatResponse(true, {
                refund: refund,
                transaction: transaction
            }, `Refund of ${this.formatCurrency(refundAmount)} processed successfully`);
            
        } catch (error) {
            this.handleError(error, 'cash refund');
            return this.formatResponse(false, null, error.message);
        }
    }

    // ========== HELPER METHODS ==========
    calculateChangeDenominations(amount) {
        const denominations = this.getConfig('denominations').sort((a, b) => b - a);
        let remaining = Math.round(amount);
        const result = {};
        
        for (const denom of denominations) {
            if (remaining >= denom) {
                const count = Math.floor(remaining / denom);
                result[denom] = count;
                remaining -= count * denom;
            }
        }
        
        if (remaining > 0) {
            result.remaining = remaining;
        }
        
        return result;
    }

    generateInvoiceNumber() {
        const date = new Date();
        const year = date.getFullYear().toString().slice(-2);
        const month = (date.getMonth() + 1).toString().padStart(2, '0');
        const day = date.getDate().toString().padStart(2, '0');
        const random = Math.floor(Math.random() * 10000).toString().padStart(4, '0');
        return `CASH${year}${month}${day}${random}`;
    }

    generateReceipt(transaction) {
        return {
            receiptNo: `RCPT-${transaction.id}`,
            date: new Date(transaction.timestamp).toLocaleString(),
            amount: transaction.amount,
            received: transaction.received,
            change: transaction.change,
            method: this.methodName,
            items: transaction.items,
            customer: transaction.customer,
            notes: transaction.notes,
            footer: 'Thank you for your purchase!'
        };
    }

    async saveTransaction(transaction) {
        try {
            // Save to local storage for offline use
            const transactions = JSON.parse(localStorage.getItem('cash_transactions') || '[]');
            transactions.push(transaction);
            localStorage.setItem('cash_transactions', JSON.stringify(transactions));
            
            // Send to server
            const result = await this.callAPI('/api/payments/cash/save', { transaction });
            
            if (result.success) {
                transaction.serverId = result.transactionId;
                this.updateLocalTransaction(transaction.id, transaction);
            }
            
            return result;
            
        } catch (error) {
            console.warn('Failed to save transaction to server:', error.message);
            // Still return success since we saved locally
            return { success: true, offline: true };
        }
    }

    async getTransaction(transactionId) {
        // Try local storage first
        const transactions = JSON.parse(localStorage.getItem('cash_transactions') || '[]');
        let transaction = transactions.find(t => t.id === transactionId);
        
        // Try server if not found locally
        if (!transaction) {
            try {
                const result = await this.callAPI('/api/payments/cash/get', { transactionId });
                if (result.success && result.transaction) {
                    transaction = result.transaction;
                }
            } catch (error) {
                console.warn('Failed to fetch transaction from server:', error.message);
            }
        }
        
        return transaction;
    }

    updateLocalTransaction(transactionId, updates) {
        const transactions = JSON.parse(localStorage.getItem('cash_transactions') || '[]');
        const index = transactions.findIndex(t => t.id === transactionId);
        
        if (index > -1) {
            transactions[index] = { ...transactions[index], ...updates };
            localStorage.setItem('cash_transactions', JSON.stringify(transactions));
        }
    }

    async saveRefund(refund) {
        try {
            const refunds = JSON.parse(localStorage.getItem('cash_refunds') || '[]');
            refunds.push(refund);
            localStorage.setItem('cash_refunds', JSON.stringify(refunds));
            
            await this.callAPI('/api/payments/cash/refund', { refund });
            
        } catch (error) {
            console.warn('Failed to save refund to server:', error.message);
        }
    }

    // ========== CASH MANAGEMENT ==========
    async getCashDrawerStatus() {
        try {
            const response = await this.callAPI('/api/cash-drawer/status');
            return response.data || {
                total: 0,
                denominations: {},
                lastCounted: null,
                expected: 0,
                discrepancy: 0
            };
        } catch (error) {
            console.warn('Failed to get cash drawer status:', error.message);
            return null;
        }
    }

    async recordCashMovement(type, amount, notes = '') {
        const movement = {
            id: `MOVEMENT_${Date.now()}`,
            type: type, // 'add', 'remove', 'float', 'withdrawal'
            amount: this.validateAmount(amount),
            notes: notes,
            timestamp: new Date().toISOString(),
            user: data.userId || 'system'
        };
        
        try {
            await this.callAPI('/api/cash-drawer/movement', { movement });
            this.logTransaction('cash_movement', movement);
            return movement;
        } catch (error) {
            this.handleError(error, 'cash movement');
            throw error;
        }
    }

    // ========== CASH FLOAT CALCULATION ==========
    calculateOptimalFloat(transactions, expectedBusiness = 'normal') {
        const hourlyPatterns = {};
        const dayOfWeekPatterns = {};
        
        // Analyze transaction patterns
        transactions.forEach(tx => {
            const date = new Date(tx.timestamp);
            const hour = date.getHours();
            const day = date.getDay();
            
            hourlyPatterns[hour] = (hourlyPatterns[hour] || 0) + tx.amount;
            dayOfWeekPatterns[day] = (dayOfWeekPatterns[day] || 0) + tx.amount;
        });
        
        // Calculate optimal float based on patterns
        let optimalFloat = 5000; // Default
        
        if (expectedBusiness === 'busy') {
            optimalFloat = 10000;
        } else if (expectedBusiness === 'slow') {
            optimalFloat = 2000;
        }
        
        return {
            optimalFloat: optimalFloat,
            hourlyPatterns: hourlyPatterns,
            dayOfWeekPatterns: dayOfWeekPatterns,
            recommendation: this.getFloatRecommendation(optimalFloat)
        };
    }

    getFloatRecommendation(amount) {
        const denominations = this.getConfig('denominations');
        let remaining = amount;
        const breakdown = {};
        
        for (const denom of denominations.sort((a, b) => b - a)) {
            if (remaining >= denom) {
                const count = Math.floor(remaining / denom);
                breakdown[denom] = count;
                remaining -= count * denom;
            }
        }
        
        return {
            total: amount,
            breakdown: breakdown,
            remaining: remaining,
            denominations: Object.keys(breakdown).map(d => ({
                value: parseInt(d),
                count: breakdown[d],
                total: parseInt(d) * breakdown[d]
            }))
        };
    }
}

export default CashProcessor;