// =========================================================
// eTIMS Integration Module - Production Grade v2.0
// =========================================================

class ETIMSIntegration {
    static version = '2.0.0';
    static credentials = null;
    static isConfigured = false;
    static pendingSubmissions = [];
    
    // Configuration
    static config = {
        enabled: true,
        autoSubmit: true,
        maxRetries: 3,
        retryDelay: 5000,
        timeout: 30000,
        apiVersion: 'v2',
        endpoint: '/api/etims'
    };
    
    /**
     * Initialize eTIMS module
     */
    static async init() {
        console.log(`🔐 eTIMS Integration v${this.version} initializing...`);
        
        try {
            // Load configuration
            await this.loadConfig();
            
            // Load credentials
            await this.loadCredentials();
            
            // Setup event listeners
            this.setupEventListeners();
            
            // Start pending submissions processor
            this.startPendingProcessor();
            
            console.log('✅ eTIMS Integration initialized');
            
            return true;
            
        } catch (error) {
            console.error('❌ eTIMS initialization failed:', error);
            this.showStatus('eTIMS integration failed to initialize', 'warning');
            return false;
        }
    }
    
    /**
     * Load configuration
     */
    static async loadConfig() {
        try {
            const response = await fetch('/api/etims/config');
            if (response.ok) {
                const data = await response.json();
                this.config = { ...this.config, ...data.config };
            }
            
            // Load from localStorage as fallback
            const savedConfig = localStorage.getItem('etims_config');
            if (savedConfig) {
                this.config = { ...this.config, ...JSON.parse(savedConfig) };
            }
            
        } catch (error) {
            console.warn('Failed to load eTIMS config:', error);
        }
    }
    
    /**
     * Load credentials
     */
    static async loadCredentials() {
        try {
            const response = await fetch('/api/etims/credentials/active');
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.credentials) {
                    this.credentials = data.credentials;
                    this.isConfigured = true;
                    console.log('🔐 eTIMS credentials loaded');
                    this.triggerEvent('etims:configured', { credentials: data.credentials });
                }
            }
        } catch (error) {
            console.warn('Failed to load eTIMS credentials:', error);
        }
    }
    
    /**
     * Setup event listeners
     */
    static setupEventListeners() {
        // Listen for receipt events
        document.addEventListener('receipt:generated', (e) => {
            if (this.config.autoSubmit) {
                setTimeout(() => this.submitReceipt(e.detail.receipt), 1000);
            }
        });
        
        // Listen for manual submission requests
        document.addEventListener('etims:submit', (e) => {
            this.submitReceipt(e.detail);
        });
        
        // Network status changes
        window.addEventListener('online', () => {
            this.retryPendingSubmissions();
        });
    }
    
    /**
     * Submit receipt to eTIMS
     */
    static async submitReceipt(receiptData) {
        if (!this.isConfigured) {
            console.log('eTIMS submission skipped - not configured');
            return { success: false, error: 'eTIMS not configured' };
        }
        
        console.log('📤 Submitting to eTIMS:', receiptData.invoice_no);
        
        try {
            // Prepare eTIMS data
            const etimsData = this.prepareETIMSData(receiptData);
            
            // Submit to server
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), this.config.timeout);
            
            const response = await fetch(`${this.config.endpoint}/submit`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                },
                body: JSON.stringify(etimsData),
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            if (!response.ok) {
                throw new Error(`eTIMS submission failed: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.message || 'eTIMS submission failed');
            }
            
            console.log('✅ eTIMS submission successful');
            this.showStatus('Receipt submitted to KRA eTIMS', 'success');
            
            // Update receipt data
            const updatedReceipt = {
                ...receiptData,
                etims: {
                    submitted: true,
                    submission_id: data.submission_id,
                    qr_code: data.qr_code,
                    verification_url: data.verification_url,
                    submitted_at: new Date().toISOString(),
                    status: 'submitted'
                }
            };
            
            // Trigger events
            this.triggerEvent('etims:submitted', {
                receipt: updatedReceipt,
                response: data
            });
            
            return {
                success: true,
                data: data,
                receipt: updatedReceipt
            };
            
        } catch (error) {
            console.error('eTIMS submission error:', error);
            
            // Save for retry
            this.saveForRetry(receiptData, error);
            
            // Show appropriate message
            if (error.name === 'AbortError') {
                this.showStatus('eTIMS submission timeout', 'warning');
            } else {
                this.showStatus('eTIMS submission failed', 'warning');
            }
            
            return {
                success: false,
                error: error.message,
                receipt: receiptData
            };
        }
    }
    
    /**
     * Prepare eTIMS data from receipt
     */
    static prepareETIMSData(receiptData) {
        const store = receiptData.store;
        const customer = receiptData.customer;
        const vatInfo = receiptData.vat_info || {};
        
        // Prepare items for eTIMS
        const items = receiptData.items.map(item => ({
            itemCode: item.sku || `ITEM${item.id}`,
            itemName: item.name.substring(0, 100),
            qty: item.quantity,
            unitPrice: item.unit_price,
            taxRate: item.vat_included ? (item.tax_rate || 16) : 0,
            taxAmount: item.tax_amount || 0,
            discount: 0,
            itemTotal: item.total || 0,
            itemClassificationCode: "1" // 1: Goods, 2: Services
        }));
        
        // Calculate totals
        const totalTax = items.reduce((sum, item) => sum + (item.taxAmount || 0), 0);
        const totalAmount = items.reduce((sum, item) => sum + (item.itemTotal || 0), 0);
        
        // Determine tax type
        let taxType = "01"; // 01: VAT
        if (vatInfo.is_exempted) {
            taxType = "04"; // 04: Exempt
        } else if (vatInfo.is_zero_rated) {
            taxType = "02"; // 02: Zero-rated
        }
        
        // Determine supply type
        let supplyTypeCode = "01"; // 01: Goods
        if (receiptData.items.some(item => item.is_service)) {
            supplyTypeCode = "02"; // 02: Services
        }
        
        return {
            header: {
                tin: store.pin || "",
                bhfId: store.branch_code || "001",
                dvcSrlNo: receiptData.transaction_id || `DVC${Date.now()}`,
                invoiceNo: receiptData.invoice_no,
                invoiceType: "01", // 01: Invoice
                invoiceDate: receiptData.date.replace(/-/g, '') || new Date().toISOString().split('T')[0].replace(/-/g, ''),
                invoiceTime: (receiptData.time || new Date().toTimeString().split(' ')[0]).replace(/:/g, ''),
                customerName: customer.name.substring(0, 100),
                customerTin: customer.pin || "",
                customerMobileNum: customer.phone || "",
                customerAddress: customer.address?.substring(0, 200) || "",
                taxType: taxType,
                supplyTypeCode: supplyTypeCode,
                paymentType: this.getETIMSPaymentType(receiptData.payment?.method || 'cash'),
                remark: ""
            },
            items: items,
            payment: {
                paymentMethod: receiptData.payment?.method?.toUpperCase() || 'CASH',
                paymentAmount: receiptData.payment?.amount_paid || receiptData.totals?.grand_total || 0,
                currency: "KES"
            },
            totals: {
                totalTax: totalTax,
                totalAmount: totalAmount,
                discount: receiptData.totals?.discount || 0,
                serviceCharge: 0,
                grandTotal: receiptData.totals?.grand_total || totalAmount
            },
            metadata: {
                receiptId: receiptData.id,
                localInvoiceNo: receiptData.invoice_no,
                customerId: customer.id,
                storeId: store.id,
                posSessionId: window.posState?.sessionId
            }
        };
    }
    
    /**
     * Get eTIMS payment type code
     */
    static getETIMSPaymentType(paymentMethod) {
        const paymentTypes = {
            'cash': '01',
            'mpesa': '02',
            'card': '03',
            'credit': '04',
            'bank': '05',
            'multiple': '06',
            'cheque': '07',
            'mobile': '02'
        };
        return paymentTypes[paymentMethod.toLowerCase()] || '01';
    }
    
    /**
     * Save receipt for retry
     */
    static saveForRetry(receiptData, error) {
        try {
            let pending = JSON.parse(localStorage.getItem('etims_pending_submissions') || '[]');
            
            const pendingItem = {
                receipt: receiptData,
                error: error.message,
                retryCount: 0,
                maxRetries: this.config.maxRetries,
                lastAttempt: new Date().toISOString(),
                nextRetry: new Date(Date.now() + this.config.retryDelay).toISOString()
            };
            
            pending.push(pendingItem);
            
            // Keep only last 50 pending submissions
            if (pending.length > 50) {
                pending = pending.slice(-50);
            }
            
            localStorage.setItem('etims_pending_submissions', JSON.stringify(pending));
            
            console.log('💾 Saved for retry:', receiptData.invoice_no);
            
        } catch (error) {
            console.error('Failed to save for retry:', error);
        }
    }
    
    /**
     * Start pending submissions processor
     */
    static startPendingProcessor() {
        setInterval(() => {
            this.processPendingSubmissions();
        }, 10000); // Check every 10 seconds
    }
    
    /**
     * Process pending submissions
     */
    static async processPendingSubmissions() {
        if (!this.isConfigured || !navigator.onLine) return;
        
        try {
            const pending = JSON.parse(localStorage.getItem('etims_pending_submissions') || '[]');
            if (pending.length === 0) return;
            
            const now = new Date();
            const toRetry = pending.filter(item => 
                new Date(item.nextRetry) <= now && 
                item.retryCount < item.maxRetries
            );
            
            for (const item of toRetry) {
                try {
                    console.log(`🔄 Retrying eTIMS submission: ${item.receipt.invoice_no}`);
                    
                    const result = await this.submitReceipt(item.receipt);
                    
                    if (result.success) {
                        // Remove from pending
                        const updated = pending.filter(p => p.receipt.invoice_no !== item.receipt.invoice_no);
                        localStorage.setItem('etims_pending_submissions', JSON.stringify(updated));
                    } else {
                        // Update retry count
                        item.retryCount++;
                        item.lastAttempt = new Date().toISOString();
                        item.nextRetry = new Date(Date.now() + this.config.retryDelay).toISOString();
                        item.error = result.error;
                        
                        // Save updated item
                        const updated = pending.map(p => 
                            p.receipt.invoice_no === item.receipt.invoice_no ? item : p
                        );
                        localStorage.setItem('etims_pending_submissions', JSON.stringify(updated));
                    }
                    
                    // Wait between retries
                    await new Promise(resolve => setTimeout(resolve, 1000));
                    
                } catch (error) {
                    console.error('Retry failed:', error);
                }
            }
            
        } catch (error) {
            console.error('Pending processor error:', error);
        }
    }
    
    /**
     * Retry all pending submissions
     */
    static async retryPendingSubmissions() {
        if (!navigator.onLine) return;
        
        console.log('🔄 Retrying all pending eTIMS submissions...');
        await this.processPendingSubmissions();
    }
    
    /**
     * Get submission status
     */
    static async getStatus(submissionId) {
        try {
            const response = await fetch(`${this.config.endpoint}/status/${submissionId}`);
            if (response.ok) {
                return await response.json();
            }
            return { success: false, error: 'Status check failed' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Verify eTIMS connection
     */
    static async verifyConnection() {
        try {
            const response = await fetch(`${this.config.endpoint}/test-connection`);
            if (response.ok) {
                const data = await response.json();
                return data;
            }
            return { success: false, error: 'Connection test failed' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Get pending submissions count
     */
    static getPendingCount() {
        try {
            const pending = JSON.parse(localStorage.getItem('etims_pending_submissions') || '[]');
            return pending.length;
        } catch (error) {
            return 0;
        }
    }
    
    /**
     * Clear pending submissions
     */
    static clearPendingSubmissions() {
        try {
            localStorage.removeItem('etims_pending_submissions');
            this.showStatus('Pending submissions cleared', 'success');
            return true;
        } catch (error) {
            this.showStatus('Failed to clear pending submissions', 'error');
            return false;
        }
    }
    
    /**
     * Export pending submissions
     */
    static exportPendingSubmissions() {
        try {
            const pending = JSON.parse(localStorage.getItem('etims_pending_submissions') || '[]');
            const dataStr = JSON.stringify(pending, null, 2);
            const dataBlob = new Blob([dataStr], { type: 'application/json' });
            
            const downloadUrl = URL.createObjectURL(dataBlob);
            const a = document.createElement('a');
            a.href = downloadUrl;
            a.download = `etims-pending-${new Date().toISOString().split('T')[0]}.json`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(downloadUrl);
            
            this.showStatus('Pending submissions exported', 'success');
            
        } catch (error) {
            console.error('Export failed:', error);
            this.showStatus('Export failed', 'error');
        }
    }
    
    /**
     * Show status message
     */
    static showStatus(message, type = 'info') {
        if (window.POSUtils?.showToast) {
            window.POSUtils.showToast(message, type);
        } else {
            console.log(`[eTIMS:${type.toUpperCase()}] ${message}`);
        }
    }
    
    /**
     * Trigger custom event
     */
    static triggerEvent(eventName, detail = {}) {
        const event = new CustomEvent(eventName, { detail });
        document.dispatchEvent(event);
    }
}

// Global initialization
setTimeout(() => {
    ETIMSIntegration.init().catch(error => {
        console.error('eTIMS module failed to initialize:', error);
    });
}, 3000);

// Global exports
window.ETIMSIntegration = ETIMSIntegration;
window.submitToETIMS = (receiptData) => ETIMSIntegration.submitReceipt(receiptData);
window.retryETIMSSubmissions = () => ETIMSIntegration.retryPendingSubmissions();
window.getETIMSStatus = (submissionId) => ETIMSIntegration.getStatus(submissionId);
window.testETIMSConnection = () => ETIMSIntegration.verifyConnection();

console.log('✅ eTIMS Integration module loaded');