// POS Discount Management Module
// =========================================================

class POSDiscounts {
    static init() {
        this.setupEventListeners();
    }
    
    static setupEventListeners() {
        const discountModal = document.getElementById('discountModal');
        if (discountModal) {
            discountModal.addEventListener('shown.bs.modal', () => {
                this.updateSummary();
                this.loadHistory();
                const discountValueInput = document.getElementById('discountValueInput');
                if (discountValueInput) discountValueInput.focus();
            });
            
            // Discount type change
            document.querySelectorAll('input[name="discountType"]').forEach(radio => {
                radio.addEventListener('change', () => this.toggleType());
            });
            
            // Discount value input
            const discountValueInput = document.getElementById('discountValueInput');
            if (discountValueInput) {
                discountValueInput.addEventListener('input', () => this.updatePreview());
            }
            
            // Discount reason change
            const discountReason = document.getElementById('discountReason');
            if (discountReason) {
                discountReason.addEventListener('change', function() {
                    const customReason = document.getElementById('customReason');
                    if (customReason) {
                        customReason.style.display = this.value === 'other' ? 'block' : 'none';
                    }
                });
            }
        }
    }
    
    static showModal() {
        const modal = new bootstrap.Modal(document.getElementById('discountModal'));
        modal.show();
    }
    
    static updateSummary() {
        const total = window.posState.getCartTotal();
        const itemCount = window.posState.getItemCount();
        
        const currentSaleTotal = document.getElementById('currentSaleTotal');
        const currentItemCount = document.getElementById('currentItemCount');
        const originalTotal = document.getElementById('originalTotal');
        
        if (currentSaleTotal) currentSaleTotal.textContent = `KES ${total.toFixed(2)}`;
        if (currentItemCount) currentItemCount.textContent = itemCount;
        if (originalTotal) originalTotal.textContent = `KES ${total.toFixed(2)}`;
        
        // Update preview if discount exists
        if (window.posState.currentDiscount > 0) {
            this.updatePreview();
        }
    }
    
    static toggleType() {
        const discountType = document.querySelector('input[name="discountType"]:checked');
        const prefix = document.getElementById('discountPrefix');
        const suffix = document.getElementById('discountSuffix');
        const hint = document.getElementById('discountHint');
        
        if (!discountType || !prefix || !suffix || !hint) return;
        
        if (discountType.value === 'amount') {
            prefix.textContent = 'KES';
            suffix.classList.add('d-none');
            hint.textContent = 'Enter discount amount in KES';
        } else {
            prefix.textContent = '';
            suffix.classList.remove('d-none');
            hint.textContent = 'Enter discount percentage (0-100)';
        }
        
        const discountValueInput = document.getElementById('discountValueInput');
        if (discountValueInput) discountValueInput.value = '';
        
        this.updatePreview();
    }
    
    static updatePreview() {
        const discountValueInput = document.getElementById('discountValueInput');
        const discountPreview = document.getElementById('discountPreview');
        
        if (!discountValueInput || !discountPreview) return;
        
        const valueInput = discountValueInput.value;
        if (!valueInput) {
            discountPreview.style.display = 'none';
            return;
        }
        
        const value = parseFloat(valueInput);
        if (isNaN(value) || value <= 0) {
            discountPreview.style.display = 'none';
            return;
        }
        
        const discountType = document.querySelector('input[name="discountType"]:checked');
        if (!discountType) return;
        
        const originalTotal = window.posState.getCartTotal();
        let discountAmount = 0;
        
        if (discountType.value === 'amount') {
            discountAmount = Math.min(value, originalTotal);
        } else {
            discountAmount = originalTotal * (Math.min(value, 100) / 100);
        }
        
        const newTotal = originalTotal - discountAmount;
        const savingsPercent = ((discountAmount / originalTotal) * 100).toFixed(1);
        
        // Update preview
        discountPreview.style.display = 'block';
        
        const discountAmountPreview = document.getElementById('discountAmountPreview');
        const newTotalEl = document.getElementById('newTotal');
        const savingsPercentEl = document.getElementById('savingsPercent');
        
        if (discountAmountPreview) discountAmountPreview.textContent = `-KES ${discountAmount.toFixed(2)}`;
        if (newTotalEl) newTotalEl.textContent = `KES ${newTotal.toFixed(2)}`;
        if (savingsPercentEl) savingsPercentEl.textContent = `${savingsPercent}%`;
    }
    
    static applyQuick(value) {
        const total = window.posState.getCartTotal();
        
        if (value <= 100) {
            // Percentage discount
            const discountPercent = document.getElementById('discountPercent');
            if (discountPercent) discountPercent.checked = true;
            this.toggleType();
            const discountValueInput = document.getElementById('discountValueInput');
            if (discountValueInput) discountValueInput.value = value;
        } else {
            // Fixed amount discount
            const discountAmount = document.getElementById('discountAmount');
            if (discountAmount) discountAmount.checked = true;
            this.toggleType();
            const discountValueInput = document.getElementById('discountValueInput');
            if (discountValueInput) discountValueInput.value = value;
        }
        
        this.updatePreview();
    }
    
    static apply() {
        const discountValueInput = document.getElementById('discountValueInput');
        const discountReason = document.getElementById('discountReason');
        const customReasonInput = document.getElementById('customReasonInput');
        
        if (!discountValueInput) {
            POSUtils.showToast('Please enter a discount value', 'error');
            return;
        }
        
        const valueInput = discountValueInput.value;
        if (!valueInput) {
            POSUtils.showToast('Please enter a discount value', 'error');
            return;
        }
        
        const value = parseFloat(valueInput);
        if (isNaN(value) || value <= 0) {
            POSUtils.showToast('Please enter a valid discount value', 'error');
            return;
        }
        
        const discountType = document.querySelector('input[name="discountType"]:checked');
        if (!discountType) return;
        
        const total = window.posState.getCartTotal();
        let discountAmount = 0;
        
        if (discountType.value === 'amount') {
            discountAmount = Math.min(value, total);
        } else {
            if (value > 100) {
                POSUtils.showToast('Discount percentage cannot exceed 100%', 'error');
                return;
            }
            discountAmount = total * (value / 100);
        }
        
        // Apply discount
        window.posState.currentDiscount = discountAmount;
        POSCart.updateDisplay();
        
        // Save discount reason
        const reasonValue = discountReason?.value || '';
        const customReasonValue = customReasonInput?.value || '';
        const discountReasonFinal = reasonValue === 'other' ? customReasonValue : reasonValue;
        
        if (discountReasonFinal) {
            this.saveToHistory(discountAmount, discountReasonFinal, discountType.value);
        }
        
        // Close modal
        const modalElement = document.getElementById('discountModal');
        if (modalElement) {
            const modal = bootstrap.Modal.getInstance(modalElement);
            if (modal) modal.hide();
        }
        
        POSUtils.showToast(`✅ Discount applied: -KES ${discountAmount.toFixed(2)}`, 'success');
    }
    
    static remove() {
        window.posState.currentDiscount = 0;
        POSCart.updateDisplay();
        
        const modalElement = document.getElementById('discountModal');
        if (modalElement) {
            const modal = bootstrap.Modal.getInstance(modalElement);
            if (modal) modal.hide();
        }
        
        POSUtils.showToast('Discount removed', 'info');
    }
    
    static saveToHistory(amount, reason, type) {
        const history = JSON.parse(localStorage.getItem('discountHistory') || '[]');
        history.unshift({
            amount: amount,
            reason: reason,
            type: type,
            date: new Date().toISOString(),
            total: window.posState.getCartTotal() + amount // Add discount back to get original total
        });
        
        if (history.length > 10) {
            history.pop();
        }
        
        localStorage.setItem('discountHistory', JSON.stringify(history));
    }
    
    static loadHistory() {
        const history = JSON.parse(localStorage.getItem('discountHistory') || '[]');
        const container = document.getElementById('discountHistory');
        const discountHistorySection = document.getElementById('discountHistorySection');
        
        if (!container || !discountHistorySection) return;
        
        if (history.length === 0) {
            discountHistorySection.style.display = 'none';
            return;
        }
        
        discountHistorySection.style.display = 'block';
        
        let html = '';
        history.forEach(item => {
            const discountType = item.type === 'amount' ? 'KES' : '%';
            const discountValue = item.type === 'amount' ? 
                `-KES ${item.amount.toFixed(2)}` : 
                `${((item.amount / item.total) * 100).toFixed(1)}%`;
            
            html += `
                <div class="list-group-item">
                    <div class="discount-history-item">
                        <div>
                            <div class="fw-bold">${discountValue}</div>
                            <div class="discount-history-reason">${item.reason}</div>
                        </div>
                        <div class="text-muted small">
                            ${new Date(item.date).toLocaleDateString()}
                        </div>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
    }
}

// Export for global access
window.POSDiscounts = POSDiscounts;
window.showDiscountModal = () => POSDiscounts.showModal();
window.applyQuickDiscount = (value) => POSDiscounts.applyQuick(value);
window.applyDiscount = () => POSDiscounts.apply();
window.removeDiscount = () => POSDiscounts.remove();
window.toggleDiscountType = () => POSDiscounts.toggleType();
window.loadDiscountHistory = () => POSDiscounts.loadHistory();