// public/js/pos/core/monitor.js - FIXED
class POSMonitor {
    constructor() {
        this.metrics = {
            startupTime: null,
            moduleLoadTimes: {},
            transactionCount: 0,
            errorCount: 0,
            lastTransactionTime: null
        };
        
        this.startTime = performance.now();
        this.setupMonitoring();
    }
    
    setupMonitoring() {
        // Track module load times
        const originalSetTimeout = window.setTimeout;
        window.setTimeout = (callback, delay, ...args) => {
            const stack = new Error().stack;
            return originalSetTimeout(() => {
                try {
                    callback(...args);
                } catch (error) {
                    this.recordError(error, { type: 'setTimeout', stack });
                    throw error;
                }
            }, delay);
        };
        
        // Track fetch requests
        const originalFetch = window.fetch;
        window.fetch = (...args) => {
            const start = performance.now();
            return originalFetch(...args).then(response => {
                const duration = performance.now() - start;
                this.recordMetric('fetch_duration', duration, { url: args[0] });
                return response;
            }).catch(error => {
                this.recordError(error, { type: 'fetch', args });
                throw error;
            });
        };
        
        // Periodic health check - REMOVED process.env.NODE_ENV CHECK
        setInterval(() => this.healthCheck(), 60000);
    }
    
    recordMetric(name, value, tags = {}) {
        this.metrics[name] = this.metrics[name] || [];
        this.metrics[name].push({
            value,
            tags,
            timestamp: Date.now()
        });
        
        // Keep only last 1000 metrics
        if (this.metrics[name].length > 1000) {
            this.metrics[name].shift();
        }
    }
    
    recordError(error, context = {}) {
        this.metrics.errorCount++;
        window.POSErrorHandler?.capture(error, context);
    }
    


healthCheck() {
    const health = {
        timestamp: Date.now(),
        modulesLoaded: Object.keys(window).filter(key => 
            key.startsWith('POS') || key === 'posState'
        ).length,
        cartItems: window.posState?.cart?.length || 0,
        memory: performance.memory ? {
            usedMB: Math.round(performance.memory.usedJSHeapSize / 1048576),
            totalMB: Math.round(performance.memory.totalJSHeapSize / 1048576)
        } : null,
        online: navigator.onLine
    };
    
    // Log to console
    console.log('🏥 POS Health:', health);
    
    // Send to server - use GET if POST fails
    fetch('/api/pos/health', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(health)
    }).catch(() => {
        // Fallback to GET
        fetch('/api/pos/health-check')
            .catch(() => {
                // Silently fail
            });
    });
    
    return health;
}



}

// Initialize
window.POSMonitor = new POSMonitor();