<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use App\Models\User;

class PermissionSeeder extends Seeder
{
    public function run()
    {
        // Reset cached roles and permissions
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // Create permissions
        $permissions = [
            // User management permissions
            'view users',
            'create users',
            'edit users',
            'delete users',
            'export users',
            'import users',
            
            // Role management permissions
            'view roles',
            'create roles',
            'edit roles',
            'delete roles',
            
            // Profile permissions
            'view profile',
            'edit profile',
            'change password',
            
            // Product permissions
            'view products',
            'create products',
            'edit products',
            'delete products',
            
            // Sale permissions
            'view sales',
            'create sales',
            'edit sales',
            'delete sales',
            
            // Purchase permissions
            'view purchases',
            'create purchases',
            'edit purchases',
            'delete purchases',
            
            // Customer permissions
            'view customers',
            'create customers',
            'edit customers',
            'delete customers',
            
            // Supplier permissions
            'view suppliers',
            'create suppliers',
            'edit suppliers',
            'delete suppliers',
            
            // Report permissions
            'view reports',
            'generate reports',
            'export reports',
            
            // Settings permissions
            'view settings',
            'edit settings',
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(['name' => $permission, 'guard_name' => 'web']);
        }

        // Create roles
        $superAdminRole = Role::firstOrCreate(['name' => 'super-admin', 'guard_name' => 'web']);
        $adminRole = Role::firstOrCreate(['name' => 'admin', 'guard_name' => 'web']);
        $managerRole = Role::firstOrCreate(['name' => 'manager', 'guard_name' => 'web']);
        $cashierRole = Role::firstOrCreate(['name' => 'cashier', 'guard_name' => 'web']);
        $purchaserRole = Role::firstOrCreate(['name' => 'purchaser', 'guard_name' => 'web']);

        // Assign all permissions to super-admin
        $superAdminRole->givePermissionTo(Permission::all());

        // Admin permissions (all except super-admin specific)
        $adminPermissions = array_filter($permissions, function($permission) {
            return !str_contains($permission, 'super-admin');
        });
        $adminRole->givePermissionTo($adminPermissions);

        // Manager permissions
        $managerRole->givePermissionTo([
            'view users',
            'edit users',
            'view profile',
            'edit profile',
            'view products',
            'edit products',
            'view sales',
            'create sales',
            'edit sales',
            'view purchases',
            'view customers',
            'edit customers',
            'view suppliers',
            'view reports',
            'generate reports',
            'view settings',
        ]);

        // Cashier permissions
        $cashierRole->givePermissionTo([
            'view profile',
            'edit profile',
            'view products',
            'view sales',
            'create sales',
            'view customers',
            'create customers',
        ]);

        // Purchaser permissions
        $purchaserRole->givePermissionTo([
            'view profile',
            'edit profile',
            'view products',
            'view purchases',
            'create purchases',
            'view suppliers',
            'create suppliers',
        ]);

        // Assign roles to existing users
        $users = [
            ['email' => 'admin@example.com', 'role' => 'admin'],
            ['email' => 'manager@example.com', 'role' => 'manager'],
            ['email' => 'cashier@example.com', 'role' => 'cashier'],
            ['email' => 'purchaser@example.com', 'role' => 'purchaser'],
        ];

        foreach ($users as $userData) {
            $user = User::where('email', $userData['email'])->first();
            if ($user) {
                $user->assignRole($userData['role']);
            }
        }

        $this->command->info('Permissions and roles seeded successfully!');
    }
}