<?php
// database/migrations/2025_12_21_000000_create_printers_table.php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // First drop existing tables if they exist
        Schema::dropIfExists('printer_logs');
        Schema::dropIfExists('printer_jobs');
        Schema::dropIfExists('printers');
        
        Schema::create('printers', function (Blueprint $table) {
            $table->id();
            
            // Basic Information
            $table->string('name');
            $table->string('alias')->nullable();
            $table->string('description')->nullable();
            
            // Connection Settings
            $table->string('connection_type')->default('network');
            $table->string('ip_address')->nullable();
            $table->integer('port')->default(9100);
            $table->string('device_path')->nullable();
            $table->string('bluetooth_address')->nullable();
            $table->string('file_path')->nullable();
            
            // Printer Configuration
            $table->string('printer_driver')->nullable();
            $table->string('paper_size')->default('80mm');
            $table->integer('chars_per_line')->default(42);
            $table->boolean('supports_graphics')->default(true);
            $table->boolean('supports_barcode')->default(true);
            $table->boolean('supports_qr_code')->default(true);
            $table->string('encoding')->default('CP850');
            
            // Print Settings
            $table->boolean('auto_cut')->default(true);
            $table->boolean('auto_open_drawer')->default(false);
            $table->integer('print_density')->default(15);
            $table->integer('print_speed')->default(3);
            $table->integer('margin_top')->default(0);
            $table->integer('margin_bottom')->default(0);
            $table->integer('margin_left')->default(0);
            $table->integer('margin_right')->default(0);
            
            // Receipt Configuration
            $table->string('header_template')->nullable();
            $table->string('footer_template')->nullable();
            $table->text('custom_css')->nullable();
            $table->boolean('print_store_logo')->default(false);
            $table->string('logo_path')->nullable();
            $table->boolean('print_barcode_on_receipt')->default(true);
            $table->boolean('print_qr_on_receipt')->default(true);
            
            // Status & Settings
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->string('status')->default('offline');
            $table->string('last_error')->nullable();
            $table->timestamp('last_used_at')->nullable();
            $table->timestamp('last_maintenance_at')->nullable();
            $table->integer('print_count')->default(0);
            $table->integer('error_count')->default(0);
            
            // Thermal Printer Specific
            $table->boolean('has_cutter')->default(true);
            $table->boolean('has_drawer_kick')->default(true);
            $table->string('drawer_pin')->default('2');
            $table->integer('drawer_pulse_time')->default(100);
            $table->string('buzzer_pin')->nullable();
            $table->boolean('has_buzzer')->default(false);
            
            // Network Printer Settings
            $table->integer('timeout')->default(30);
            $table->integer('retry_count')->default(3);
            $table->boolean('use_ssl')->default(false);
            $table->string('auth_type')->nullable();
            $table->string('username')->nullable();
            $table->string('password')->nullable();
            
            // Print Job Settings
            $table->integer('max_job_queue')->default(10);
            $table->integer('job_timeout')->default(60);
            $table->boolean('enable_queue')->default(true);
            $table->boolean('enable_retry')->default(true);
            
            // Usage Statistics
            $table->integer('total_prints')->default(0);
            $table->integer('successful_prints')->default(0);
            $table->integer('failed_prints')->default(0);
            $table->decimal('success_rate', 5, 2)->default(0);
            $table->integer('paper_used')->default(0);
            $table->timestamp('paper_last_changed')->nullable();
            $table->integer('estimated_paper_remaining')->nullable();
            
            // Printer Health
            $table->integer('temperature')->nullable();
            $table->string('head_status')->nullable();
            $table->string('paper_status')->nullable();
            $table->string('cover_status')->nullable();
            $table->string('cutter_status')->nullable();
            $table->timestamp('last_health_check')->nullable();
            
            // Location & Assignment
            $table->string('location')->nullable();
            $table->string('department')->nullable();
            $table->foreignId('assigned_to')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('terminal_id')->nullable()->constrained('terminals')->onDelete('set null');
            
            // Maintenance
            $table->integer('maintenance_interval')->default(30);
            $table->timestamp('next_maintenance_date')->nullable();
            $table->text('maintenance_notes')->nullable();
            $table->string('service_provider')->nullable();
            $table->string('service_contact')->nullable();
            
            // Cost & Warranty
            $table->decimal('purchase_cost', 10, 2)->nullable();
            $table->timestamp('purchase_date')->nullable();
            $table->string('warranty_period')->nullable();
            $table->timestamp('warranty_expiry')->nullable();
            $table->decimal('operational_cost_per_print', 8, 4)->nullable();
            
            // Metadata
            $table->json('capabilities')->nullable();
            $table->json('settings')->nullable();
            $table->json('custom_commands')->nullable();
            
            // Audit
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index('connection_type');
            $table->index('is_default');
            $table->index('is_active');
            $table->index('status');
            $table->index('location');
            $table->index('department');
            $table->index(['is_active', 'status']);
        });
        
        Schema::create('printer_jobs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('printer_id')->constrained()->onDelete('cascade');
            $table->string('job_type');
            $table->string('job_status')->default('pending');
            $table->text('content');
            $table->integer('priority')->default(5);
            $table->integer('attempts')->default(0);
            $table->text('error_message')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamp('processed_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamps();
            
            $table->index('job_status');
            $table->index('priority');
            $table->index(['printer_id', 'job_status']);
            $table->index('created_at');
        });
        
        Schema::create('printer_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('printer_id')->constrained()->onDelete('cascade');
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('action');
            $table->text('description')->nullable();
            $table->json('details')->nullable();
            $table->string('ip_address')->nullable();
            $table->string('user_agent')->nullable();
            $table->timestamps();
            
            $table->index(['printer_id', 'action']);
            $table->index('created_at');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('printer_logs');
        Schema::dropIfExists('printer_jobs');
        Schema::dropIfExists('printers');
    }
};