<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        // First, let's check what columns exist
        Schema::table('categories', function (Blueprint $table) {
            // Add slug if it doesn't exist
            if (!Schema::hasColumn('categories', 'slug')) {
                $table->string('slug')->nullable()->after('name');
            }
            
            // Add is_active column (this is what your model expects)
            if (!Schema::hasColumn('categories', 'is_active')) {
                $table->boolean('is_active')->default(true)->after('description');
            }
            
            // Add parent_id if it doesn't exist
            if (!Schema::hasColumn('categories', 'parent_id')) {
                $table->foreignId('parent_id')->nullable()->after('is_active')->constrained('categories')->onDelete('set null');
            }
            
            // Add color if it doesn't exist
            if (!Schema::hasColumn('categories', 'color')) {
                $table->string('color', 7)->nullable()->after('parent_id');
            }
            
            // Add image if it doesn't exist
            if (!Schema::hasColumn('categories', 'image')) {
                $table->string('image')->nullable()->after('color');
            }
            
            // Add sort_order if it doesn't exist
            if (!Schema::hasColumn('categories', 'sort_order')) {
                $table->integer('sort_order')->default(0)->after('image');
            }
            
            // Remove status column if it exists (use is_active instead)
            if (Schema::hasColumn('categories', 'status')) {
                $table->dropColumn('status');
            }
            
            // Add indexes
            $table->index('slug');
            $table->index('is_active');
            $table->index('parent_id');
            $table->index('sort_order');
        });
    }

    public function down(): void
    {
        Schema::table('categories', function (Blueprint $table) {
            // Drop columns if they exist
            $columns = ['slug', 'is_active', 'parent_id', 'color', 'image', 'sort_order'];
            
            foreach ($columns as $column) {
                if (Schema::hasColumn('categories', $column)) {
                    $table->dropColumn($column);
                }
            }
            
            // Drop indexes
            $indexes = ['slug', 'is_active', 'parent_id', 'sort_order'];
            foreach ($indexes as $index) {
                if (Schema::hasIndex('categories', $index)) {
                    $table->dropIndex(['categories_' . $index . '_index']);
                }
            }
        });
    }
};