<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('brands', function (Blueprint $table) {
            // Add slug column with unique constraint
            $table->string('slug')->unique()->nullable()->after('name');
            
            // Add description column
            $table->text('description')->nullable()->after('slug');
            
            // Add logo column
            $table->string('logo')->nullable()->after('description');
            
            // Add contact information columns
            $table->string('website')->nullable()->after('logo');
            $table->string('contact_email')->nullable()->after('website');
            $table->string('contact_phone')->nullable()->after('contact_email');
            
            // Add status column
            $table->enum('status', ['active', 'inactive'])->default('active')->after('contact_phone');
            
            // Add country information
            $table->string('country')->nullable()->after('status');
            $table->integer('established_year')->nullable()->after('country');
            
            // Add SEO columns
            $table->string('meta_title')->nullable()->after('established_year');
            $table->text('meta_description')->nullable()->after('meta_title');
            $table->text('meta_keywords')->nullable()->after('meta_description');
            
            // Add featured and order columns
            $table->boolean('is_featured')->default(false)->after('meta_keywords');
            $table->integer('order')->default(0)->after('is_featured');
            
            // Add indexes
            $table->index('slug');
            $table->index('status');
            $table->index('is_featured');
            $table->index('order');
        });
        
        // Create pivot table for brand-category relationship
        if (!Schema::hasTable('brand_category')) {
            Schema::create('brand_category', function (Blueprint $table) {
                $table->id();
                $table->foreignId('brand_id')->constrained()->onDelete('cascade');
                $table->foreignId('category_id')->constrained()->onDelete('cascade');
                $table->timestamps();
                
                $table->unique(['brand_id', 'category_id']);
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('brand_category');
        
        Schema::table('brands', function (Blueprint $table) {
            // Remove indexes first
            $table->dropIndex(['slug']);
            $table->dropIndex(['status']);
            $table->dropIndex(['is_featured']);
            $table->dropIndex(['order']);
            
            // Remove columns
            $table->dropColumn([
                'slug',
                'description',
                'logo',
                'website',
                'contact_email',
                'contact_phone',
                'status',
                'country',
                'established_year',
                'meta_title',
                'meta_description',
                'meta_keywords',
                'is_featured',
                'order'
            ]);
        });
    }
};