<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Unit extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'abbreviation',
        'description',
        'status',
        'conversion_rate',
        'base_unit_id',
        'is_base_unit',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'conversion_rate' => 'decimal:4',
        'is_base_unit' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($unit) {
            if ($unit->is_base_unit) {
                $unit->conversion_rate = 1;
                $unit->base_unit_id = null;
            }
        });

        static::updating(function ($unit) {
            if ($unit->isDirty('is_base_unit') && $unit->is_base_unit) {
                $unit->conversion_rate = 1;
                $unit->base_unit_id = null;
            }
        });
    }

    /**
     * Get the base unit.
     */
    public function baseUnit()
    {
        return $this->belongsTo(Unit::class, 'base_unit_id');
    }

    /**
     * Get the derived units.
     */
    public function derivedUnits()
    {
        return $this->hasMany(Unit::class, 'base_unit_id');
    }

    /**
     * Get the products using this unit.
     */
    public function products()
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Scope a query to only include active units.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope a query to only include base units.
     */
    public function scopeBaseUnits($query)
    {
        return $query->where('is_base_unit', true);
    }

    /**
     * Scope a query to only include derived units.
     */
    public function scopeDerivedUnits($query)
    {
        return $query->where('is_base_unit', false);
    }

    /**
     * Convert quantity from this unit to target unit.
     */
    public function convertTo($targetUnitId, $quantity)
    {
        if ($this->id == $targetUnitId) {
            return $quantity;
        }

        $targetUnit = self::find($targetUnitId);
        
        if (!$targetUnit) {
            return null;
        }

        // If this is a base unit
        if ($this->is_base_unit) {
            if ($targetUnit->base_unit_id == $this->id) {
                return $quantity * $targetUnit->conversion_rate;
            }
            return null;
        }

        // If this is a derived unit
        if ($targetUnit->is_base_unit) {
            if ($this->base_unit_id == $targetUnit->id) {
                return $quantity / $this->conversion_rate;
            }
            return null;
        }

        // Both are derived units with same base
        if ($this->base_unit_id == $targetUnit->base_unit_id) {
            return ($quantity * $this->conversion_rate) / $targetUnit->conversion_rate;
        }

        return null;
    }

    /**
     * Get the name with abbreviation.
     */
    public function getFullNameAttribute()
    {
        return $this->name . ' (' . $this->abbreviation . ')';
    }

    /**
     * Get the conversion formula.
     */
    public function getConversionFormulaAttribute()
    {
        if ($this->is_base_unit) {
            return "1 {$this->abbreviation} = 1 {$this->abbreviation}";
        }

        if ($this->baseUnit) {
            $rate = number_format($this->conversion_rate, 4);
            return "1 {$this->baseUnit->abbreviation} = {$rate} {$this->abbreviation}";
        }

        return "No conversion available";
    }

    /**
     * Get the status badge HTML.
     */
    public function getStatusBadgeAttribute()
    {
        $status = $this->status;
        $badgeClass = $status == 'active' ? 'success' : 'danger';
        
        return '<span class="badge bg-' . $badgeClass . '">' . ucfirst($status) . '</span>';
    }

    /**
     * Get the type badge HTML.
     */
    public function getTypeBadgeAttribute()
    {
        $type = $this->is_base_unit ? 'base' : 'derived';
        $badgeClass = $type == 'base' ? 'info' : 'secondary';
        
        return '<span class="badge bg-' . $badgeClass . '">' . ucfirst($type) . ' Unit</span>';
    }

    /**
     * Check if unit can be deleted.
     */
    public function getCanDeleteAttribute()
    {
        return $this->products()->count() == 0 && 
               $this->derivedUnits()->count() == 0;
    }

    /**
     * Get related units (same base unit).
     */
    public function getRelatedUnitsAttribute()
    {
        if ($this->is_base_unit) {
            return $this->derivedUnits;
        }

        return self::where('base_unit_id', $this->base_unit_id)
            ->where('id', '!=', $this->id)
            ->get();
    }
}