<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Services\NumberGeneratorService;


class Sale extends Model
{
    use HasFactory;

    protected $fillable = [
        'unified_number',
        'invoice_no',
        'receipt_no',
        'qr_code_path',
        'customer_id',
        'user_id',
        'subtotal',
        'tax_total',
        'discount',
        'grand_total',
        'status',
        'is_vat_applied',
        'etims_invoice_number',
        'etims_qr_code',
        'etims_response',
        'etims_sent_at',
        'etims_status',
        'cash_received',
        'mpesa_received',
        'card_received',
        'bank_received',
        'credit_amount',
        'customer_change',
        'payment_method',
        'transaction_id',
        'payment_status',
        'notes',
        'sale_date',
    ];

    protected $casts = [
        'unified_number' => 'string',
        'etims_response' => 'array',
        'etims_sent_at' => 'datetime',
        'sale_date' => 'datetime',
        'is_vat_applied' => 'boolean',
        'subtotal' => 'decimal:2',
        'tax_total' => 'decimal:2',
        'discount' => 'decimal:2',
        'grand_total' => 'decimal:2',
        'cash_received' => 'decimal:2',
        'mpesa_received' => 'decimal:2',
        'card_received' => 'decimal:2',
        'bank_received' => 'decimal:2',
        'credit_amount' => 'decimal:2',
        'customer_change' => 'decimal:2',
        'zero_rated' => 'boolean',
    ];

    /**
     * Get the customer that owns the sale.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Get the user (cashier) that made the sale.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the items for the sale.
     */
    public function items(): HasMany
    {
        return $this->hasMany(SaleItem::class);
    }

    /**
     * Get the M-Pesa transactions for the sale.
     */
    public function mpesaTransactions(): HasMany
    {
        return $this->hasMany(MpesaTransaction::class);
    }

    /**
     * Scope a query to only include sales from today.
     */
    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    /**
     * Scope a query to only include completed sales.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope a query to only include paid sales.
     */
    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    /**
     * Get the invoice number with fallback to receipt number.
     */
    public function getInvoiceNumberAttribute(): string
    {
        return $this->invoice_no ?? $this->receipt_no;
    }

    /**
     * Check if sale has VAT applied.
     */
    public function hasVat(): bool
    {
        return $this->is_vat_applied && $this->tax_total > 0;
    }

    /**
     * Get the total items in the sale.
     */
    public function getTotalItemsAttribute(): int
    {
        return $this->items->sum('qty');
    }


/**
 * Get the display number (invoice first, then receipt)
 */


/**
 * Check if sale has an invoice number
 */
public function getHasInvoiceAttribute(): bool
{
    return !empty($this->invoice_no);
}

/**
 * Check if sale has a receipt number
 */
public function getHasReceiptAttribute(): bool
{
    return !empty($this->receipt_no);
}

/**
 * Get display number (invoice first, receipt fallback)
 */
public function getDisplayNumberAttribute(): string
{
    return $this->invoice_no ?? $this->receipt_no ?? 'N/A';
}

/**
 * Get formatted number based on type
 */
public function getFormattedNumber(string $type = 'display'): string
{
    $numberGenerator = app(NumberGeneratorService::class);
    
    switch ($type) {
        case 'invoice':
            return $this->invoice_no ?? 'N/A';
        case 'receipt':
            return $this->receipt_no ?? 'N/A';
        case 'both':
            $invoice = $this->invoice_no ? "Invoice: {$this->invoice_no}" : '';
            $receipt = $this->receipt_no ? "Receipt: {$this->receipt_no}" : '';
            return trim("{$invoice} {$receipt}");
        case 'short':
            return substr($this->display_number, -8); // Last 8 chars
        default:
            return $this->display_number;
    }
}

/**
 * Generate numbers for this sale
 */
public function generateNumbers(): self
{
    if (!$this->invoice_no) {
        $numberGenerator = app(NumberGeneratorService::class);
        $numbers = $numberGenerator->generateForSale();
        
        $this->invoice_no = $numbers['invoice'];
        $this->receipt_no = $numbers['receipt'];
        $this->save();
    }
    
    return $this;
}

/**
 * Scope for sales with invoice numbers
 */
public function scopeHasInvoice($query)
{
    return $query->whereNotNull('invoice_no');
}

/**
 * Scope for sales with receipt numbers
 */
public function scopeHasReceipt($query)
{
    return $query->whereNotNull('receipt_no');
}

/**
 * Get numbers in array format
 */
public function getNumbersAttribute(): array
{
    return [
        'invoice' => $this->invoice_no,
        'receipt' => $this->receipt_no,
        'display' => $this->display_number,
        'has_invoice' => !empty($this->invoice_no),
        'has_receipt' => !empty($this->receipt_no)
    ];
}
/**
 * Calculate VAT based on customer status
 */
public function calculateVatForCustomer(Customer $customer = null): array
{
    if (!$customer) {
        // No customer or walk-in customer - apply VAT
        return $this->calculateStandardVat();
    }
    
    // Check customer's VAT status
    switch ($customer->current_vat_status) {
        case 'exempted':
            return [
                'taxable_amount' => 0,
                'tax_amount' => 0,
                'total_with_vat' => $this->subtotal - $this->discount,
                'vat_rate' => 0,
                'vat_status' => 'exempted'
            ];
            
        case 'zero_rated':
            return [
                'taxable_amount' => $this->subtotal - $this->discount,
                'tax_amount' => 0,
                'total_with_vat' => $this->subtotal - $this->discount,
                'vat_rate' => 0,
                'vat_status' => 'zero_rated'
            ];
            
        case 'vatable':
        default:
            return $this->calculateStandardVat();
    }
}

/**
 * Calculate standard VAT (16%)
 */
protected function calculateStandardVat(): array
{
    $taxableAmount = $this->subtotal - $this->discount;
    $taxAmount = ($taxableAmount * 16) / 100;
    
    return [
        'taxable_amount' => $taxableAmount,
        'tax_amount' => $taxAmount,
        'total_with_vat' => $taxableAmount + $taxAmount,
        'vat_rate' => 16,
        'vat_status' => 'vatable'
    ];
}

// In app/Models/Sale.php - Add these methods:

/**
 * Calculate VAT for sale with customer status and product-specific rates
 */
public function calculateItemVAT(Customer $customer = null, $product, $quantity): array
{
    // Get customer status
    $customerStatus = $customer ? $customer->current_vat_status : 'vatable';
    
    // Get product tax rate (default 16%)
    $taxRate = $product->tax_rate ?? 16;
    
    // Calculate base price
    $priceExcludingVAT = $product->price_ex_tax ?? $product->price ?? 0;
    $itemTotalExcludingVAT = $priceExcludingVAT * $quantity;
    
    // Apply customer status
    if ($customerStatus === 'exempted') {
        return [
            'price_ex_tax' => $priceExcludingVAT,
            'tax_rate' => 0,
            'tax_amount' => 0,
            'total' => $itemTotalExcludingVAT,
            'vat_status' => 'exempted'
        ];
    }
    
    if ($customerStatus === 'zero_rated') {
        return [
            'price_ex_tax' => $priceExcludingVAT,
            'tax_rate' => 0,
            'tax_amount' => 0,
            'total' => $itemTotalExcludingVAT,
            'vat_status' => 'zero_rated'
        ];
    }
    
    // VATable customer
    $taxAmount = ($itemTotalExcludingVAT * $taxRate) / 100;
    $total = $itemTotalExcludingVAT + $taxAmount;
    
    return [
        'price_ex_tax' => $priceExcludingVAT,
        'tax_rate' => $taxRate,
        'tax_amount' => $taxAmount,
        'total' => $total,
        'vat_status' => 'vatable'
    ];
}

/**
 * Calculate sale totals with proper VAT handling
 */
public function calculateSaleTotals(array $items, Customer $customer = null): array
{
    $subtotal = 0;
    $taxTotal = 0;
    $taxableAmount = 0;
    $nonTaxableAmount = 0;
    $vatByRate = [];
    
    foreach ($items as $item) {
        $product = Product::find($item['product_id']);
        $vatInfo = $this->calculateItemVAT($customer, $product, $item['quantity']);
        
        $subtotal += $vatInfo['total'];
        
        if ($vatInfo['tax_rate'] > 0) {
            $taxTotal += $vatInfo['tax_amount'];
            $taxableAmount += $vatInfo['price_ex_tax'] * $item['quantity'];
            
            // Group by tax rate
            $rateKey = (string)$vatInfo['tax_rate'];
            if (!isset($vatByRate[$rateKey])) {
                $vatByRate[$rateKey] = [
                    'rate' => $vatInfo['tax_rate'],
                    'taxable_amount' => 0,
                    'tax_amount' => 0
                ];
            }
            $vatByRate[$rateKey]['taxable_amount'] += $vatInfo['price_ex_tax'] * $item['quantity'];
            $vatByRate[$rateKey]['tax_amount'] += $vatInfo['tax_amount'];
        } else {
            $nonTaxableAmount += $vatInfo['total'];
        }
    }
    
    $grandTotal = $subtotal;
    
    return [
        'subtotal' => $subtotal,
        'tax_total' => $taxTotal,
        'taxable_amount' => $taxableAmount,
        'non_taxable_amount' => $nonTaxableAmount,
        'grand_total' => $grandTotal,
        'vat_by_rate' => $vatByRate,
        'customer_vat_status' => $customer ? $customer->current_vat_status : 'vatable'
    ];
}

/**
     * Scope to find by unified number
     */
    public function scopeByUnifiedNumber($query, $unifiedNumber)
    {
        return $query->where('unified_number', $unifiedNumber)
            ->orWhere('invoice_no', $unifiedNumber)
            ->orWhere('receipt_no', $unifiedNumber);
    }
    
    /**
     * Generate unified number for new sale
     */
    public static function generateUnifiedNumber(): string
    {
        return UnifiedNumberSequence::getNextNumber();
    }
    
    /**
     * Get today's sales count
     */
    public static function getTodaySalesCount(): int
    {
        $today = now()->format('Y-m-d');
        return self::whereDate('created_at', $today)->count();
    }


}