<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class MpesaTransaction extends Model
{
    protected $fillable = [
        'transaction_id',
        'reference',
        'phone',
        'amount',
        'expected_amount',
        'transaction_date',
        'status',
        'description',
        'first_name',
        'middle_name',
        'last_name',
        'sale_id',
        'customer_id',
        'raw_response'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'expected_amount' => 'decimal:2',
        'transaction_date' => 'datetime',
        'raw_response' => 'array'
    ];

    public function sale()
    {
        return $this->belongsTo(Sale::class);
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    // Scope for pending transactions
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    // Scope for completed transactions
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    // Scope for unmatched transactions
    public function scopeUnmatched($query)
    {
        return $query->whereNull('sale_id')->where('status', 'completed');
    }

    // Check if transaction matches expected amount
    public function matchesExpectedAmount(): bool
    {
        return abs($this->amount - $this->expected_amount) < 0.01;
    }

    // Get full name from M-Pesa response
    public function getFullNameAttribute(): string
    {
        $names = array_filter([$this->first_name, $this->middle_name, $this->last_name]);
        return implode(' ', $names);
    }
}