<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Crypt;

class MpesaCredential extends Model
{
    use SoftDeletes;

    // Explicitly define the table name
    protected $table = 'mpesa_credentials';

    protected $fillable = [
        'name',
        'business_shortcode',
        'consumer_key',
        'consumer_secret',
        'passkey',
        'initiator_name',
        'initiator_password',
        'security_credential',
        'environment',
        'callback_url',
        'is_active',
        'last_success',
        'last_failure',
        'notes'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'last_success' => 'datetime',
        'last_failure' => 'datetime',
    ];

    protected $hidden = [
        'consumer_key',
        'consumer_secret',
        'passkey',
        'initiator_password',
        'security_credential'
    ];

    protected $appends = [
        'shortcode_masked',
        'environment_display'
    ];

    /**
     * Encrypt sensitive data before saving.
     */
    public function setConsumerKeyAttribute($value)
    {
        $this->attributes['consumer_key'] = Crypt::encryptString($value);
    }

    public function setConsumerSecretAttribute($value)
    {
        $this->attributes['consumer_secret'] = Crypt::encryptString($value);
    }

    public function setPasskeyAttribute($value)
    {
        $this->attributes['passkey'] = Crypt::encryptString($value);
    }

    public function setInitiatorPasswordAttribute($value)
    {
        $this->attributes['initiator_password'] = Crypt::encryptString($value);
    }

    public function setSecurityCredentialAttribute($value)
    {
        $this->attributes['security_credential'] = Crypt::decryptString($value);
    }

    /**
     * Decrypt sensitive data when needed.
     */
    public function getDecryptedConsumerKey()
    {
        try {
            return Crypt::decryptString($this->attributes['consumer_key']);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDecryptedConsumerSecret()
    {
        try {
            return Crypt::decryptString($this->attributes['consumer_secret']);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDecryptedPasskey()
    {
        try {
            return Crypt::decryptString($this->attributes['passkey']);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDecryptedInitiatorPassword()
    {
        try {
            return Crypt::decryptString($this->attributes['initiator_password']);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDecryptedSecurityCredential()
    {
        try {
            return Crypt::decryptString($this->attributes['security_credential']);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Accessors.
     */
    public function getShortcodeMaskedAttribute()
    {
        $shortcode = $this->business_shortcode;
        return substr($shortcode, 0, 2) . '****' . substr($shortcode, -2);
    }

    public function getEnvironmentDisplayAttribute()
    {
        return $this->environment === 'production' ? 'Production' : 'Sandbox';
    }

    /**
     * Scope active credentials.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
}