<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Category extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'parent_id',
        'is_active',
        'color',
        'image',
        'sort_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'image_url',
        'breadcrumb',
        'has_subcategories',
        'total_products_recursive'
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            // Generate slug from name if not provided
            if (empty($category->slug)) {
                $category->slug = Str::slug($category->name);
                
                // Make slug unique
                $originalSlug = $category->slug;
                $count = 1;
                while (static::where('slug', $category->slug)->exists()) {
                    $category->slug = $originalSlug . '-' . $count++;
                }
            }
        });

        static::updating(function ($category) {
            // Regenerate slug if name changed
            if ($category->isDirty('name') && empty($category->slug)) {
                $category->slug = Str::slug($category->name);
                
                // Make slug unique (excluding current category)
                $originalSlug = $category->slug;
                $count = 1;
                while (static::where('slug', $category->slug)
                        ->where('id', '!=', $category->id)
                        ->exists()) {
                    $category->slug = $originalSlug . '-' . $count++;
                }
            }
        });
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    public function parent()
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    public function children()
    {
        return $this->hasMany(Category::class, 'parent_id')->orderBy('sort_order')->orderBy('name');
    }

    /**
     * Scope a query to only include active categories.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include root categories (no parent).
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Get the category image URL.
     */
    public function getImageUrlAttribute()
    {
        if ($this->image) {
            // Check if it's a full URL
            if (filter_var($this->image, FILTER_VALIDATE_URL)) {
                return $this->image;
            }
            
            // Check if it's a storage path
            if (strpos($this->image, 'storage/') === 0) {
                return asset($this->image);
            }
            
            // Check if file exists in storage
            if (file_exists(storage_path('app/public/' . $this->image))) {
                return asset('storage/' . $this->image);
            }
        }
        
        // Return a placeholder based on category color or name
        $colors = [
            '#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#ffeaa7',
            '#dda0dd', '#98d8c8', '#f7b731', '#eb3b5a', '#26de81'
        ];
        
        $color = $this->color ?: ($colors[crc32($this->name) % count($colors)] ?? '#667eea');
        
        // Remove # from color for URL
        $backgroundColor = ltrim($color, '#');
        
        return "https://ui-avatars.com/api/?name=" . urlencode(substr($this->name, 0, 2)) . 
               "&background=" . $backgroundColor . 
               "&color=fff&size=300&bold=true&font-size=0.5";
    }

    /**
     * Get the full category path (breadcrumb).
     */
    public function getPathAttribute()
    {
        $path = [];
        $category = $this;
        
        while ($category) {
            array_unshift($path, $category);
            $category = $category->parent;
        }
        
        return collect($path);
    }

    /**
     * Get the category breadcrumb as string.
     */
    public function getBreadcrumbAttribute()
    {
        return $this->path->pluck('name')->implode(' > ');
    }

    /**
     * Check if category has subcategories.
     */
    public function getHasSubcategoriesAttribute()
    {
        return $this->children()->exists();
    }

    /**
     * Get total products including subcategories.
     */
    public function getTotalProductsRecursiveAttribute()
    {
        $total = $this->products()->count();
        
        // Use a recursive function to get all child products
        $getChildProducts = function($category) use (&$getChildProducts) {
            $subtotal = 0;
            foreach ($category->children as $child) {
                $subtotal += $child->products()->count();
                $subtotal += $getChildProducts($child);
            }
            return $subtotal;
        };
        
        return $total + $getChildProducts($this);
    }

    /**
     * Get the formatted created_at date.
     */
    public function getFormattedCreatedAtAttribute()
    {
        return $this->created_at->format('M d, Y');
    }

    /**
     * Get the formatted updated_at date.
     */
    public function getFormattedUpdatedAtAttribute()
    {
        return $this->updated_at->format('M d, Y');
    }

    /**
     * Check if category can be deleted.
     */
    public function getCanDeleteAttribute()
    {
        return $this->products()->count() === 0 && $this->children()->count() === 0;
    }
}