<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Brand extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'slug',
        'description',
        'logo',
        'website',
        'contact_email',
        'contact_phone',
        'status',
        'country',
        'established_year',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'is_featured',
        'order',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'established_year' => 'integer',
        'is_featured' => 'boolean',
        'order' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($brand) {
            if (empty($brand->slug)) {
                $brand->slug = \Illuminate\Support\Str::slug($brand->name);
            }
        });

        static::updating(function ($brand) {
            if ($brand->isDirty('name') && empty($brand->slug)) {
                $brand->slug = \Illuminate\Support\Str::slug($brand->name);
            }
        });
    }

    /**
     * Get the products for the brand.
     */
    public function products()
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the categories for the brand.
     */
    public function categories()
    {
        return $this->belongsToMany(Category::class, 'brand_category');
    }

    /**
     * Scope a query to only include active brands.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope a query to only include featured brands.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to order by custom order.
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('order')->orderBy('name');
    }

    /**
     * Get the logo URL.
     */
    public function getLogoUrlAttribute()
    {
        if ($this->logo) {
            return asset('storage/brands/' . $this->logo);
        }
        
        return asset('images/default-brand.png');
    }

    /**
     * Get the products count.
     */
    public function getProductsCountAttribute()
    {
        return $this->products()->count();
    }

    /**
     * Get the active products count.
     */
    public function getActiveProductsCountAttribute()
    {
        return $this->products()->where('status', 'active')->count();
    }

    /**
     * Get the display name with optional country.
     */
    public function getDisplayNameAttribute()
    {
        if ($this->country) {
            return $this->name . ' (' . $this->country . ')';
        }
        return $this->name;
    }

    /**
     * Get the status badge HTML.
     */
    public function getStatusBadgeAttribute()
    {
        $status = $this->status;
        $badgeClass = $status == 'active' ? 'success' : 'danger';
        
        return '<span class="badge bg-' . $badgeClass . '">' . ucfirst($status) . '</span>';
    }

    /**
     * Check if brand has logo.
     */
    public function getHasLogoAttribute()
    {
        return !empty($this->logo);
    }
}