<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BatchStockMovement extends Model
{
    use HasFactory;

    protected $fillable = [
        'batch_id', 'product_id',
        'movement_type', 'quantity', 'quantity_before',
        'reference_type', 'reference_id', 'reference_number',
        'reason', 'notes', 'user_id',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'quantity_before' => 'integer',
    ];

    protected $appends = [
        'quantity_after',
        'formatted_quantity',
    ];

    /**
     * Relationships
     */
    public function batch(): BelongsTo
    {
        return $this->belongsTo(ProductBatch::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Accessors
     */
    public function getQuantityAfterAttribute(): int
    {
        return $this->quantity_before + $this->quantity;
    }

    public function getFormattedQuantityAttribute(): string
    {
        $sign = $this->quantity >= 0 ? '+' : '';
        return $sign . $this->quantity;
    }

    /**
     * Scopes
     */
    public function scopeIncoming($query)
    {
        return $query->where('movement_type', 'in');
    }

    public function scopeOutgoing($query)
    {
        return $query->where('movement_type', 'out');
    }
}