<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Sale;
use Illuminate\Http\Request;

class VatReportController extends Controller
{
    /**
     * Generate VAT summary report
     */
    public function summary(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth());
        $endDate = $request->input('end_date', now()->endOfMonth());

        $sales = Sale::whereBetween('sale_date', [$startDate, $endDate])->get();

        // Categorize sales by VAT status
        $vatableSales = $sales->where('is_vat_applied', 1);
        $nonVatableSales = $sales->where('is_vat_applied', 0);

        // Group by customer type
        $customerTypes = [
            'vatable' => [],
            'exempted' => [],
            'zero_rated' => []
        ];

        foreach ($sales as $sale) {
            if ($sale->customer) {
                $status = $sale->customer->current_vat_status;
                if (isset($customerTypes[$status])) {
                    $customerTypes[$status][] = $sale;
                } else {
                    $customerTypes['vatable'][] = $sale; // fallback
                }
            } else {
                $customerTypes['vatable'][] = $sale; // Walk-in customers
            }
        }

        return response()->json([
            'period' => [
                'start' => $startDate,
                'end' => $endDate,
            ],
            'summary' => [
                'total_sales' => $sales->sum('grand_total'),
                'total_vatable_sales' => $vatableSales->sum('taxable_amount'),
                'total_vat_collected' => $vatableSales->sum('tax_amount'),
                'total_exempted_sales' => $nonVatableSales->where('zero_rated', 0)->sum('grand_total'),
                'total_zero_rated_sales' => $nonVatableSales->where('zero_rated', 1)->sum('grand_total'),
                'number_of_invoices' => $sales->count(),
            ],
            'by_customer_type' => [
                'vatable' => [
                    'count' => count($customerTypes['vatable']),
                    'amount' => collect($customerTypes['vatable'])->sum('grand_total'),
                    'vat_amount' => collect($customerTypes['vatable'])->sum('tax_amount'),
                ],
                'exempted' => [
                    'count' => count($customerTypes['exempted']),
                    'amount' => collect($customerTypes['exempted'])->sum('grand_total'),
                ],
                'zero_rated' => [
                    'count' => count($customerTypes['zero_rated']),
                    'amount' => collect($customerTypes['zero_rated'])->sum('grand_total'),
                ]
            ],
            'customers' => [
                'total' => Customer::count(),
                'vatable' => Customer::vatable()->count(),
                'exempted' => Customer::exempted()->count(),
                'zero_rated' => Customer::zeroRated()->count(),
                'with_valid_exemption' => Customer::withValidExemption()->count(),
                'with_valid_zero_rating' => Customer::withValidZeroRating()->count(),
            ]
        ]);
    }

    /**
     * Generate detailed VAT report
     */
    public function detailed(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth());
        $endDate = $request->input('end_date', now()->endOfMonth());

        $sales = Sale::with(['customer', 'items.product'])
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->orderBy('sale_date')
            ->get();

        $report = [];

        foreach ($sales as $sale) {
            $vatStatus = $sale->is_vat_applied ? 'vatable' :
                        ($sale->zero_rated ? 'zero_rated' : 'exempted');

            $report[] = [
                'date' => $sale->sale_date instanceof \Carbon\Carbon
                    ? $sale->sale_date->format('Y-m-d')
                    : $sale->sale_date,
                'invoice_no' => $sale->invoice_no,
                'customer' => $sale->customer ? $sale->customer->name : 'Walk-in',
                'customer_vat_status' => $sale->customer ? $sale->customer->current_vat_status : 'vatable',
                'vat_status' => $vatStatus,
                'subtotal' => $sale->subtotal,
                'discount' => $sale->discount,
                'taxable_amount' => $sale->taxable_amount,
                'tax_amount' => $sale->tax_amount,
                'grand_total' => $sale->grand_total,
                'items' => $sale->items->map(function($item) {
                    return [
                        'name' => optional($item->product)->name,
                        'quantity' => $item->qty,
                        'price' => $item->price,
                        'total' => $item->total,
                    ];
                })
            ];
        }

        return response()->json([
            'period' => ['start' => $startDate, 'end' => $endDate],
            'report' => $report
        ]);
    }

    /**
     * Get customers with expiring VAT certificates
     */
    public function expiringCertificates(Request $request)
    {
        $daysBefore = $request->input('days_before', 30);

        $expiring = Customer::getExpiringCertificates($daysBefore);

        return response()->json([
            'expiring_exemptions' => $expiring['exemptions']->map(function($customer) {
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'exemption_number' => $customer->vat_exemption_number,
                    'certificate_no' => $customer->vat_exemption_certificate_no,
                    'end_date' => $customer->vat_exemption_end_date,
                    'days_remaining' => now()->diffInDays($customer->vat_exemption_end_date, false),
                ];
            }),
            'expiring_zero_rated' => $expiring['zero_rated']->map(function($customer) {
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'zero_rated_reason' => $customer->zero_rated_reason,
                    'end_date' => $customer->zero_rated_end_date,
                    'days_remaining' => now()->diffInDays($customer->zero_rated_end_date, false),
                ];
            })
        ]);
    }
}
