<?php

namespace App\Http\Controllers;

use App\Models\StoreInformation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class StoreInformationWebController extends Controller
{
    /**
     * Cache keys
     */
    const CACHE_KEY = 'store_information';
    const CACHE_TTL = 3600; // 1 hour
    
    /**
     * Display store information form
     */
    public function index()
    {
        // Try to get from database
        $store = StoreInformation::first();
        
        // If no store exists in database, use fallback config
        if (!$store) {
            $store = $this->getFallbackConfig();
        } else {
            // Ensure business_hours is an array even if it's stored as JSON string
            if ($store->business_hours && is_string($store->business_hours)) {
                $store->business_hours = json_decode($store->business_hours, true);
            }
            
            // Add is_fallback flag for consistency
            $store->is_fallback = false;
        }
        
        return view('settings.store-information', [
            'store' => $store,
            'config' => config('pos', [])
        ]);
    }

/**
 * Store or update store information from web form
 */
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'store_name'       => 'required|string|max:150',
        'logo'             => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        'logo_url'         => 'nullable|string', // Remove max:500 constraint
        'whatsapp_number'  => 'nullable|string|max:20',
        'telephone'        => 'required|string|max:20',
        'address'          => 'required|string|max:500',
        'company_pin'      => 'nullable|string|max:50',
        'email'            => 'nullable|email|max:100',
        'city'             => 'nullable|string|max:100',
        'country'          => 'nullable|string|max:100',
        'currency'         => 'nullable|string|max:10',
        'timezone'         => 'nullable|string|max:50',
        'vat_number'       => 'nullable|string|max:50',
        'branch_code'      => 'nullable|string|max:20',
        'business_registration' => 'nullable|string|max:100',
        'kra_pin'          => 'nullable|string|max:50',
        'permit_number'    => 'nullable|string|max:100',
        'business_hours'   => 'nullable|string',
    ]);

    // Custom validation for logo_url
    $validator->sometimes('logo_url', 'max:500', function ($input) {
        // Only validate length if it's not a base64 string (starts with data:)
        return $input->logo_url && !str_starts_with($input->logo_url, 'data:image/');
    });

    $validator->sometimes('logo_url', 'url', function ($input) {
        // Only validate as URL if it's not a base64 string
        return $input->logo_url && !str_starts_with($input->logo_url, 'data:image/');
    });

    // Check if it's an AJAX request
    if ($request->ajax()) {
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Please correct the errors below.',
                'errors' => $validator->errors()
            ], 422);
        }
    } else {
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('error', 'Please correct the errors below.');
        }
    }

    try {
        // Parse business hours
        $businessHours = null;
        if ($request->filled('business_hours')) {
            $businessHours = json_decode($request->business_hours, true);
        } elseif ($request->filled('business_hours_open') && $request->filled('business_hours_close')) {
            $businessHours = [
                'open' => $request->business_hours_open,
                'close' => $request->business_hours_close
            ];
        }

        // Prepare data for update
        $data = $validator->validated();
        
        // Handle logo upload

        // Handle logo upload
$logoPath = null;
if ($request->hasFile('logo')) {
    // Delete old logo if exists
    $oldStore = StoreInformation::first();
    if ($oldStore && $oldStore->logo) {
        // Check if it's a local file (not a URL)
        if (!filter_var($oldStore->logo, FILTER_VALIDATE_URL)) {
            $oldPath = 'public/' . $oldStore->logo;
            if (Storage::exists($oldPath)) {
                Storage::delete($oldPath);
            }
        }
    }
    
    // Store new logo
    $logoPath = $request->file('logo')->store('store/logos', 'public');
    $data['logo'] = $logoPath; // This will be like 'store/logos/filename.png'
}

        elseif ($request->filled('logo_url')) {
            $logoUrl = $request->logo_url;
            
            // Check if it's a base64 string (file upload preview)
            if (str_starts_with($logoUrl, 'data:image/')) {
                // Extract the base64 data
                $imageData = explode(',', $logoUrl, 2)[1];
                $imageData = base64_decode($imageData);
                
                // Get image info
                $imageInfo = getimagesizefromstring($imageData);
                $extension = image_type_to_extension($imageInfo[2], false);
                
                // Generate unique filename
                $filename = 'logo_' . time() . '.' . $extension;
                $path = 'store/logos/' . $filename;
                
                // Store the file
                Storage::put('public/' . $path, $imageData);
                
                // Delete old logo if exists
                $oldStore = StoreInformation::first();
                if ($oldStore && $oldStore->logo) {
                    if (!filter_var($oldStore->logo, FILTER_VALIDATE_URL)) {
                        if (Storage::exists('public/' . $oldStore->logo)) {
                            Storage::delete('public/' . $oldStore->logo);
                        }
                    }
                }
                
                $data['logo'] = $path;
            } else {
                // It's a regular URL, store it as is
                $data['logo'] = $logoUrl;
            }
        }
        
        // Remove logo from data array if empty to avoid overwriting with null
        if (empty($data['logo']) && !$request->hasFile('logo') && !$request->filled('logo_url')) {
            unset($data['logo']);
        }
        
        // Remove logo_url field as it's not in database
        unset($data['logo_url']);
        
        if ($businessHours) {
            $data['business_hours'] = $businessHours;
        }

        // Use updateOrCreate with id=1 to ensure single record
        $store = StoreInformation::updateOrCreate(
            ['id' => 1],
            $data
        );

        // Clear all store-related cache
        $this->clearStoreCache();
        
        // Log the update
        Log::info('Store information updated via web', [
            'user_id' => auth()->id(),
            'store_id' => $store->id,
            'changes' => array_keys($validator->validated())
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Store information saved successfully!',
                'redirect' => route('settings.store-information.index')
            ]);
        } else {
            return redirect()->route('settings.store-information.index')
                ->with('success', 'Store information saved successfully!')
                ->with('store', $store);
        }

    } catch (\Exception $e) {
        Log::error('Failed to save store information via web: ' . $e->getMessage());
        
        $errorMessage = 'Failed to save store information. Please try again.';
        
        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => $errorMessage,
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        } else {
            return redirect()->back()
                ->withInput()
                ->with('error', $errorMessage);
        }
    }
}

    /**
     * Get store health status
     */
    public function health()
    {
        try {
            $hasStore = StoreInformation::exists();
            $store = StoreInformation::first();
            
            $cacheStatus = [
                'general' => Cache::has(self::CACHE_KEY) ? 'cached' : 'not_cached',
                'web' => Cache::has(self::CACHE_KEY . '_web') ? 'cached' : 'not_cached',
                'pos' => Cache::has(self::CACHE_KEY . '_pos') ? 'cached' : 'not_cached',
                'receipt' => Cache::has(self::CACHE_KEY . '_receipt') ? 'cached' : 'not_cached',
            ];

            $fallbackConfig = $this->getFallbackConfig();
            
            return response()->json([
                'success' => true,
                'health' => [
                    'database' => $hasStore ? 'has_data' : 'empty',
                    'cache' => $cacheStatus,
                    'cache_ttl' => self::CACHE_TTL,
                    'using_fallback' => !$hasStore,
                    'last_updated' => $store ? $store->updated_at->toISOString() : null,
                    'store_record_exists' => $hasStore,
                    'config_source' => 'config/pos.php',
                    'fallback_config' => $fallbackConfig
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get health status',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get store configuration
     */
    public function config()
    {
        try {
            $config = config('pos', []);
            
            return response()->json([
                'success' => true,
                'config' => [
                    'store' => $config['store'] ?? [],
                    'invoice' => $config['invoice'] ?? [],
                    'vat' => $config['vat'] ?? [],
                    'currency' => $config['currency'] ?? [],
                    'receipts' => $config['receipts'] ?? [],
                    'cache_ttl' => self::CACHE_TTL,
                ],
                'environment' => config('app.env'),
                'app_name' => config('app.name'),
                'version' => config('app.version', '1.0.0')
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load configuration',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get receipt settings
     */
    public function receiptSettings()
    {
        try {
            $storeInfo = Cache::remember(self::CACHE_KEY . '_receipt', self::CACHE_TTL, function() {
                $store = StoreInformation::first();
                
                if (!$store) {
                    return $this->getFallbackConfig();
                }
                
                return $store;
            });
            
            // Get full URL for logo if it's stored locally
            if (isset($storeInfo->logo) && $storeInfo->logo) {
                if (!filter_var($storeInfo->logo, FILTER_VALIDATE_URL)) {
                    $storeInfo->logo_url = asset('storage/' . $storeInfo->logo);
                } else {
                    $storeInfo->logo_url = $storeInfo->logo;
                }
            }
            
            return response()->json([
                'success' => true,
                'settings' => [
                    'store' => $storeInfo,
                    'receipt' => [
                        'footer_message' => config('pos.receipts.footer_message', 'Thank you for your business!'),
                        'include_qr' => config('pos.receipts.include_qr', true),
                        'include_vat' => config('pos.receipts.include_vat', true),
                        'paper_size' => config('pos.receipts.paper_size', 'A4'),
                        'print_mode' => config('pos.receipts.print_mode', 'html'),
                        'template' => config('pos.receipts.template', 'default'),
                        'auto_print' => config('pos.receipts.auto_print', true),
                        'show_logo' => config('pos.receipts.show_logo', true),
                        'show_qr_code' => config('pos.receipts.show_qr_code', true),
                        'show_business_info' => config('pos.receipts.show_business_info', true),
                    ]
                ],
                'source' => $storeInfo instanceof StoreInformation ? 'database' : 'config_fallback'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Receipt settings error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load receipt settings',
                'error' => config('app.debug') ? $e->getMessage() : null,
                'settings' => $this->getEmergencyReceiptSettings()
            ]);
        }
    }

    /**
     * Clear store cache
     */
    public function clearCache()
    {
        try {
            $this->clearStoreCache();
            
            return response()->json([
                'success' => true,
                'message' => 'Store information cache cleared successfully',
                'cleared_keys' => [
                    self::CACHE_KEY,
                    self::CACHE_KEY . '_web',
                    self::CACHE_KEY . '_pos',
                    self::CACHE_KEY . '_receipt'
                ],
                'timestamp' => now()->toISOString()
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to clear store cache: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear cache'
            ], 500);
        }
    }

    /**
     * Get fallback configuration from config/pos.php
     */
    private function getFallbackConfig()
    {
        $storeConfig = config('pos.store', []);
        
        return (object)[
            'store_name' => $storeConfig['name'] ?? env('STORE_NAME', 'Kenyan Supermarket'),
            'logo' => $storeConfig['logo'] ?? env('STORE_LOGO', ''),
            'whatsapp_number' => $storeConfig['whatsapp'] ?? env('STORE_WHATSAPP', ''),
            'telephone' => $storeConfig['phone'] ?? env('STORE_PHONE', '0700 000 000'),
            'address' => $storeConfig['address'] ?? env('STORE_ADDRESS', 'Nairobi CBD'),
            'company_pin' => $storeConfig['company_pin'] ?? env('STORE_PIN', ''),
            'email' => $storeConfig['email'] ?? env('STORE_EMAIL', ''),
            'city' => $storeConfig['city'] ?? env('STORE_CITY', 'Nairobi'),
            'country' => $storeConfig['country'] ?? env('STORE_COUNTRY', 'Kenya'),
            'currency' => $storeConfig['currency'] ?? env('STORE_CURRENCY', 'KES'),
            'timezone' => $storeConfig['timezone'] ?? env('STORE_TIMEZONE', 'Africa/Nairobi'),
            'vat_number' => $storeConfig['vat_number'] ?? env('STORE_VAT_NUMBER', ''),
            'branch_code' => $storeConfig['branch_code'] ?? env('STORE_BRANCH_CODE', '001'),
            'business_registration' => $storeConfig['business_registration'] ?? env('BUSINESS_REGISTRATION', ''),
            'kra_pin' => $storeConfig['kra_pin'] ?? env('KRA_PIN', ''),
            'permit_number' => $storeConfig['permit_number'] ?? env('PERMIT_NUMBER', ''),
            'business_hours' => [
                'open' => $storeConfig['business_hours']['open'] ?? env('STORE_OPEN', '08:00'),
                'close' => $storeConfig['business_hours']['close'] ?? env('STORE_CLOSE', '20:00'),
            ],
            'is_fallback' => true,
            'config_source' => 'config/pos.php'
        ];
    }

    /**
     * Get emergency receipt settings
     */
    private function getEmergencyReceiptSettings()
    {
        return [
            'store' => $this->getFallbackConfig(),
            'receipt' => [
                'footer_message' => 'Thank you for your business!',
                'include_qr' => true,
                'include_vat' => true,
                'paper_size' => 'A4',
                'print_mode' => 'html',
                'template' => 'default',
                'auto_print' => true,
                'emergency' => true
            ]
        ];
    }

    /**
     * Clear all store-related cache
     */
    private function clearStoreCache()
    {
        Cache::forget(self::CACHE_KEY);
        Cache::forget(self::CACHE_KEY . '_web');
        Cache::forget(self::CACHE_KEY . '_pos');
        Cache::forget(self::CACHE_KEY . '_receipt');
        
        // Also clear any other cache keys that might store store info
        $cacheKeys = config('pos.cache.keys', []);
        foreach ($cacheKeys as $key) {
            if (str_contains($key, 'store')) {
                Cache::forget($key);
            }
        }
        
        // Log cache clearance
        Log::info('Store information cache cleared', [
            'user_id' => auth()->id(),
            'keys' => [
                self::CACHE_KEY,
                self::CACHE_KEY . '_web',
                self::CACHE_KEY . '_pos',
                self::CACHE_KEY . '_receipt'
            ]
        ]);
    }


    ///////

/**
 * Fix existing logo paths (run once)
 */
public function fixLogoPaths()
{
    try {
        $store = StoreInformation::first();
        
        if ($store && $store->logo) {
            // Check current logo path
            echo "Current logo in DB: " . $store->logo . "<br>";
            echo "Generated URL: " . $store->logo_url . "<br>";
            
            // Check if it's a local file
            if (!filter_var($store->logo, FILTER_VALIDATE_URL)) {
                $logoPath = $store->logo;
                
                // Remove 'store/logos/' prefix to get just filename
                $filename = str_replace('store/logos/', '', $logoPath);
                echo "Filename only: " . $filename . "<br>";
                
                // Check if file exists in different locations
                $locations = [
                    'public/store/logos/' . $filename,
                    'public/' . $logoPath,
                    'store/logos/' . $filename,
                    $filename
                ];
                
                foreach ($locations as $location) {
                    $exists = Storage::exists($location);
                    echo "Checking: '{$location}' - " . ($exists ? 'EXISTS' : 'NOT FOUND') . "<br>";
                    
                    if ($exists) {
                        // Get file size
                        $size = Storage::size($location);
                        echo "File size: " . $size . " bytes<br>";
                        
                        // Get full path
                        $fullPath = Storage::path($location);
                        echo "Full path: " . $fullPath . "<br>";
                        
                        // Get public URL
                        if (strpos($location, 'public/') === 0) {
                            $publicPath = str_replace('public/', '', $location);
                            $publicUrl = asset('storage/' . $publicPath);
                            echo "Public URL: " . $publicUrl . "<br>";
                        }
                        break;
                    }
                }
                
                // List all files in logos directory
                echo "<br>All files in storage/app/public/store/logos:<br>";
                if (Storage::exists('public/store/logos')) {
                    $files = Storage::files('public/store/logos');
                    foreach ($files as $file) {
                        echo basename($file) . "<br>";
                    }
                } else {
                    echo "Directory doesn't exist!<br>";
                }
                
                echo "<br>All files in storage/app/public:<br>";
                $allFiles = Storage::files('public');
                foreach ($allFiles as $file) {
                    echo $file . "<br>";
                }
                
            } else {
                echo "Logo is a URL.<br>";
            }
        } else {
            echo "No store or logo found.<br>";
        }
        
    } catch (\Exception $e) {
        echo "Error: " . $e->getMessage() . "<br>";
        echo "Trace: " . $e->getTraceAsString();
    }
}



}