<?php

namespace App\Http\Controllers;

use App\Models\StoreInformation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;

class StoreInformationController extends Controller
{
    /**
     * Cache keys
     */
    const CACHE_KEY = 'store_information';
    const CACHE_TTL = 3600; // 1 hour
    
    /**
     * Display store information with fallbacks from config/pos.php
     * GET /api/store/info
     */
    public function index()
    {
        // Use caching for better performance
        $storeInfo = Cache::remember(self::CACHE_KEY, self::CACHE_TTL, function() {
            $store = StoreInformation::first();
            
            if (!$store) {
                // Return fallback configuration from config/pos.php
                return $this->getFallbackConfig();
            }
            
            return $store;
        });
        
        // Format the response for API
        return response()->json([
            'success' => true,
            'data' => $this->formatStoreInfo($storeInfo),
            'source' => $storeInfo instanceof StoreInformation ? 'database' : 'config_fallback'
        ]);
    }

    /**
     * Store or update store information.
     * POST /api/store/info
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'store_name'       => 'required|string|max:150',
            'logo'             => 'nullable|string|max:500',
            'whatsapp_number'  => 'nullable|string|max:20',
            'telephone'        => 'required|string|max:20',
            'address'          => 'required|string|max:500',
            'company_pin'      => 'nullable|string|max:50',
            'email'            => 'nullable|email|max:100',
            'city'             => 'nullable|string|max:100',
            'country'          => 'nullable|string|max:100',
            'currency'         => 'nullable|string|max:10',
            'timezone'         => 'nullable|string|max:50',
            'vat_number'       => 'nullable|string|max:50',
            'branch_code'      => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Use updateOrCreate with id=1 to ensure single record
            $store = StoreInformation::updateOrCreate(
                ['id' => 1],
                $validator->validated()
            );

            // Clear all store-related cache
            $this->clearStoreCache();
            
            // Log the update
            \Log::info('Store information updated', [
                'user_id' => auth()->id(),
                'store_id' => $store->id,
                'changes' => $validator->validated()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Store information saved successfully',
                'data' => $this->formatStoreInfo($store)
            ], 201);

        } catch (\Exception $e) {
            \Log::error('Failed to save store information: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to save store information',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update store information.
     * PUT /api/store/info/{id}
     */
    public function update(Request $request, StoreInformation $storeInformation)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'store_name'       => 'sometimes|required|string|max:150',
            'logo'             => 'sometimes|nullable|string|max:500',
            'whatsapp_number'  => 'sometimes|nullable|string|max:20',
            'telephone'        => 'sometimes|required|string|max:20',
            'address'          => 'sometimes|required|string|max:500',
            'company_pin'      => 'sometimes|nullable|string|max:50',
            'email'            => 'sometimes|nullable|email|max:100',
            'city'             => 'sometimes|nullable|string|max:100',
            'country'          => 'sometimes|nullable|string|max:100',
            'currency'         => 'sometimes|nullable|string|max:10',
            'timezone'         => 'sometimes|nullable|string|max:50',
            'vat_number'       => 'sometimes|nullable|string|max:50',
            'branch_code'      => 'sometimes|nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Update only provided fields
            $storeInformation->update($validator->validated());

            // Clear all store-related cache
            $this->clearStoreCache();
            
            // Log the update
            \Log::info('Store information partially updated', [
                'user_id' => auth()->id(),
                'store_id' => $storeInformation->id,
                'changes' => $validator->validated()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Store information updated successfully',
                'data' => $this->formatStoreInfo($storeInformation)
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to update store information: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update store information',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get store information for POS system
     * GET /api/store/pos-info
     */
    public function posInfo()
    {
        try {
            $storeInfo = Cache::remember(self::CACHE_KEY . '_pos', self::CACHE_TTL, function() {
                $store = StoreInformation::first();
                
                if (!$store) {
                    return $this->getFallbackConfig();
                }
                
                return $store;
            });
            
            return response()->json([
                'success' => true,
                'store' => $this->formatForPOS($storeInfo),
                'cache_hit' => Cache::has(self::CACHE_KEY . '_pos') ? 'yes' : 'no',
                'source' => $storeInfo instanceof StoreInformation ? 'database' : 'config_fallback'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Failed to get POS store info: ' . $e->getMessage());
            
            return response()->json([
                'success' => true, // Always return success for POS to prevent crashes
                'store' => $this->getFallbackConfig(),
                'error' => config('app.debug') ? $e->getMessage() : null,
                'emergency_fallback' => true,
                'source' => 'emergency_fallback'
            ]);
        }
    }

    /**
     * Get store settings for receipt generation
     * GET /api/store/receipt-settings
     */
    public function receiptSettings()
    {
        $storeInfo = Cache::remember(self::CACHE_KEY . '_receipt', self::CACHE_TTL, function() {
            $store = StoreInformation::first();
            
            if (!$store) {
                return $this->getFallbackConfig();
            }
            
            return $store;
        });
        
        return response()->json([
            'success' => true,
            'settings' => [
                'store' => $this->formatForReceipt($storeInfo),
                'receipt' => [
                    'footer_message' => config('pos.receipts.footer_message', 'Thank you for your business!'),
                    'include_qr' => config('pos.receipts.include_qr', true),
                    'include_vat' => config('pos.receipts.include_vat', true),
                    'paper_size' => config('pos.receipts.paper_size', 'A4'),
                    'print_mode' => config('pos.receipts.print_mode', 'html'),
                    'template' => config('pos.receipts.template', 'default'),
                    'auto_print' => config('pos.receipts.auto_print', true),
                ]
            ],
            'source' => $storeInfo instanceof StoreInformation ? 'database' : 'config_fallback'
        ]);
    }

    /**
     * Clear store information cache
     * POST /api/store/clear-cache
     */
    public function clearCache()
    {
        try {
            $this->clearStoreCache();
            
            return response()->json([
                'success' => true,
                'message' => 'Store information cache cleared',
                'cleared_keys' => [
                    self::CACHE_KEY,
                    self::CACHE_KEY . '_pos',
                    self::CACHE_KEY . '_receipt'
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear cache'
            ], 500);
        }
    }

    /**
     * Get health status of store information
     * GET /api/store/health
     */
    public function health()
    {
        $hasStore = StoreInformation::exists();
        $cacheStatus = Cache::has(self::CACHE_KEY) ? 'cached' : 'not_cached';
        $posCacheStatus = Cache::has(self::CACHE_KEY . '_pos') ? 'cached' : 'not_cached';
        
        return response()->json([
            'success' => true,
            'health' => [
                'database' => $hasStore ? 'has_data' : 'empty',
                'cache' => [
                    'general' => $cacheStatus,
                    'pos' => $posCacheStatus,
                    'ttl' => self::CACHE_TTL,
                ],
                'using_fallback' => !$hasStore,
                'config_source' => 'config/pos.php',
                'fallback_config' => $this->getFallbackConfig()
            ]
        ]);
    }

    /**
     * Get store configuration for admin panel
     * GET /api/store/config
     */
    public function config()
    {
        return response()->json([
            'success' => true,
            'config' => [
                'store' => config('pos.store', []),
                'invoice' => config('pos.invoice', []),
                'vat' => config('pos.vat', []),
                'currency' => config('pos.currency', []),
                'receipts' => config('pos.receipts', []),
                'cache_ttl' => self::CACHE_TTL,
            ]
        ]);
    }

    /**
     * Format store info for API response
     */
    private function formatStoreInfo($store)
    {
        if ($store instanceof StoreInformation) {
            return [
                'id' => $store->id,
                'store_name' => $store->store_name,
                'logo' => $store->logo,
                'whatsapp_number' => $store->whatsapp_number,
                'telephone' => $store->telephone,
                'address' => $store->address,
                'company_pin' => $store->company_pin,
                'email' => $store->email ?? config('pos.store.email', ''),
                'city' => $store->city ?? config('pos.store.city', 'Nairobi'),
                'country' => $store->country ?? config('pos.store.country', 'Kenya'),
                'currency' => $store->currency ?? config('pos.store.currency', 'KES'),
                'timezone' => $store->timezone ?? config('pos.store.timezone', 'Africa/Nairobi'),
                'vat_number' => $store->vat_number ?? config('pos.store.vat_number', ''),
                'branch_code' => $store->branch_code ?? config('pos.store.branch_code', '001'),
                'business_registration' => $store->business_registration ?? config('pos.store.business_registration', ''),
                'kra_pin' => $store->kra_pin ?? config('pos.store.kra_pin', ''),
                'permit_number' => $store->permit_number ?? config('pos.store.permit_number', ''),
                'created_at' => $store->created_at,
                'updated_at' => $store->updated_at
            ];
        }
        
        // If it's fallback config
        return $store;
    }

    /**
     * Format store info for POS system
     */
    private function formatForPOS($store)
    {
        if ($store instanceof StoreInformation) {
            return [
                'name' => $store->store_name,
                'address' => $store->address,
                'phone' => $store->telephone,
                'whatsapp' => $store->whatsapp_number,
                'email' => $store->email ?? config('pos.store.email', ''),
                'pin' => $store->company_pin,
                'vat_number' => $store->vat_number ?? config('pos.store.vat_number', ''),
                'branch_code' => $store->branch_code ?? config('pos.store.branch_code', '001'),
                'city' => $store->city ?? config('pos.store.city', 'Nairobi'),
                'country' => $store->country ?? config('pos.store.country', 'Kenya'),
                'currency' => $store->currency ?? config('pos.store.currency', 'KES'),
                'timezone' => $store->timezone ?? config('pos.store.timezone', 'Africa/Nairobi'),
                'logo' => $store->logo ?? config('pos.store.logo', ''),
                'business_hours' => [
                    'open' => config('pos.store.business_hours.open', '08:00'),
                    'close' => config('pos.store.business_hours.close', '20:00'),
                ]
            ];
        }
        
        // If it's fallback config
        return $store;
    }

    /**
     * Format store info for receipts
     */
    private function formatForReceipt($store)
    {
        if ($store instanceof StoreInformation) {
            return [
                'name' => $store->store_name,
                'address' => $store->address,
                'phone' => $store->telephone,
                'email' => $store->email ?? config('pos.store.email', ''),
                'pin' => $store->company_pin,
                'vat_number' => $store->vat_number ?? config('pos.store.vat_number', ''),
                'branch_code' => $store->branch_code ?? config('pos.store.branch_code', '001'),
                'city' => $store->city ?? config('pos.store.city', 'Nairobi'),
                'country' => $store->country ?? config('pos.store.country', 'Kenya'),
                'currency' => $store->currency ?? config('pos.store.currency', 'KES'),
                'logo' => $store->logo ?? config('pos.store.logo', ''),
            ];
        }
        
        // If it's fallback config
        return $store;
    }

    /**
     * Get fallback configuration from config/pos.php
     */
    private function getFallbackConfig()
    {
        $storeConfig = config('pos.store', []);
        
        return [
            'name' => $storeConfig['name'] ?? env('STORE_NAME', 'Kenyan Supermarket'),
            'address' => $storeConfig['address'] ?? env('STORE_ADDRESS', 'Nairobi CBD'),
            'phone' => $storeConfig['phone'] ?? env('STORE_PHONE', '0700 000 000'),
            'whatsapp' => $storeConfig['whatsapp'] ?? env('STORE_WHATSAPP', ''),
            'email' => $storeConfig['email'] ?? env('STORE_EMAIL', ''),
            'pin' => $storeConfig['company_pin'] ?? env('STORE_PIN', ''),
            'vat_number' => $storeConfig['vat_number'] ?? env('STORE_VAT_NUMBER', ''),
            'branch_code' => $storeConfig['branch_code'] ?? env('STORE_BRANCH_CODE', '001'),
            'city' => $storeConfig['city'] ?? env('STORE_CITY', 'Nairobi'),
            'country' => $storeConfig['country'] ?? env('STORE_COUNTRY', 'Kenya'),
            'currency' => $storeConfig['currency'] ?? env('STORE_CURRENCY', 'KES'),
            'timezone' => $storeConfig['timezone'] ?? env('STORE_TIMEZONE', 'Africa/Nairobi'),
            'logo' => $storeConfig['logo'] ?? env('STORE_LOGO', ''),
            'business_registration' => $storeConfig['business_registration'] ?? env('BUSINESS_REGISTRATION', ''),
            'kra_pin' => $storeConfig['kra_pin'] ?? env('KRA_PIN', ''),
            'permit_number' => $storeConfig['permit_number'] ?? env('PERMIT_NUMBER', ''),
            'business_hours' => [
                'open' => $storeConfig['business_hours']['open'] ?? env('STORE_OPEN', '08:00'),
                'close' => $storeConfig['business_hours']['close'] ?? env('STORE_CLOSE', '20:00'),
            ],
            'is_fallback' => true,
            'config_source' => 'config/pos.php'
        ];
    }

    /**
     * Clear all store-related cache
     */
    private function clearStoreCache()
    {
        Cache::forget(self::CACHE_KEY);
        Cache::forget(self::CACHE_KEY . '_pos');
        Cache::forget(self::CACHE_KEY . '_receipt');
        
        // Also clear any other cache keys that might store store info
        $cacheKeys = config('pos.cache.keys', []);
        foreach ($cacheKeys as $key) {
            if (str_contains($key, 'store')) {
                Cache::forget($key);
            }
        }
    }
}