<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;
use Illuminate\Validation\Rules;

class ProfileController extends Controller
{
    /**
     * Show the form for editing user profile.
     */
    public function edit()
    {
        $user = Auth::user();
        $user->load('roles', 'loginHistory'); // Eager load relationships
        
        return view('profile.edit', compact('user'));
    }

    /**
     * Update user profile.
     */
    public function update(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'department' => ['nullable', 'string', 'max:100'],
            'position' => ['nullable', 'string', 'max:100'],
            'address' => ['nullable', 'string', 'max:255'],
            'date_of_birth' => ['nullable', 'date'],
            'gender' => ['nullable', 'in:male,female,other'],
        ]);

        try {
            $user->update($request->all());

            // Log activity
            activity()
                ->causedBy($user)
                ->performedOn($user)
                ->log('updated profile');

            return redirect()->route('profile.edit')
                ->with('success', 'Profile updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating profile: ' . $e->getMessage());
        }
    }

    /**
     * Update user password.
     */
    public function updatePassword(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        try {
            $user->update([
                'password' => Hash::make($request->password)
            ]);

            // Log activity
            activity()
                ->causedBy($user)
                ->performedOn($user)
                ->log('changed password');

            return redirect()->route('profile.edit')
                ->with('success', 'Password updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating password: ' . $e->getMessage());
        }
    }

    /**
     * Update profile image.
     */
    public function updateImage(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'profile_image' => ['required', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048']
        ]);

        try {
            // Delete old image if exists
            if ($user->profile_image) {
                Storage::disk('public')->delete('profile-images/' . $user->profile_image);
            }
            
            $image = $request->file('profile_image');
            $filename = 'user_' . $user->id . '_' . time() . '.' . $image->getClientOriginalExtension();
            
            // Create directory if it doesn't exist
            $path = 'profile-images/';
            Storage::disk('public')->makeDirectory($path);
            
            // Resize and save image
            $image = Image::make($image)->fit(300, 300)->encode();
            Storage::disk('public')->put($path . $filename, $image);
            
            $user->update(['profile_image' => $filename]);

            // Log activity
            activity()
                ->causedBy($user)
                ->performedOn($user)
                ->log('updated profile image');

            return redirect()->route('profile.edit')
                ->with('success', 'Profile image updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating profile image: ' . $e->getMessage());
        }
    }

    /**
     * Get user statistics.
     */
    public function statistics()
    {
        $user = Auth::user();
        
        $statistics = [
            'total_sales' => $user->sales()->count(),
            'total_purchases' => $user->purchases()->count(),
            'total_activities' => $user->activities()->count(),
            'last_login' => $user->last_login_at ? $user->last_login_at->diffForHumans() : 'Never',
            'login_count' => $user->loginHistory()->count(),
        ];

        return view('profile.statistics', compact('user', 'statistics'));
    }
}