<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Supplier;
use App\Models\Unit;
use App\Models\Tax;
use App\Models\ProductBatch;
use App\Models\PriceHistory;
use App\Models\BatchStockMovement;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Product::with(['category', 'brand', 'supplier', 'unit'])
                ->withCount('batches');

            // Search functionality
            if ($request->has('search')) {
                $search = $request->input('search');
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%")
                      ->orWhere('barcode', 'like', "%{$search}%")
                      ->orWhere('ref_number', 'like', "%{$search}%");
                });
            }

            // Filter by category
            if ($request->has('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            // Filter by brand
            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->brand_id);
            }

            // Filter by supplier
            if ($request->has('supplier_id')) {
                $query->where('supplier_id', $request->supplier_id);
            }

            // Filter by status
            if ($request->has('status') && $request->status != '') {
                $query->where('status', $request->status);
            }

            // Filter by stock status
            if ($request->has('stock_status') && $request->stock_status != '') {
                switch ($request->stock_status) {
                    case 'in_stock':
                        $query->where('stock', '>', 0);
                        break;
                    case 'low_stock':
                        $query->whereColumn('stock', '<=', 'minimum_stock')
                              ->where('stock', '>', 0);
                        break;
                    case 'out_of_stock':
                        $query->where('stock', '<=', 0);
                        break;
                }
            }

            // Filter by expiry tracking
            if ($request->has('has_expiry') && $request->has_expiry != '') {
                if ($request->has_expiry == '1') {
                    $query->where('has_expiry', true);
                } else {
                    $query->where('has_expiry', false);
                }
            }

            // Filter by tax status
            if ($request->has('has_vat') && $request->has_vat != '') {
                $query->where('has_vat', $request->has_vat == '1');
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            
            $allowedSort = ['name', 'sku', 'stock', 'sale_price', 'cost_price', 'created_at', 'updated_at'];
            if (in_array($sortBy, $allowedSort)) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->latest();
            }

            $products = $query->paginate(20)->withQueryString();

            // Statistics for dashboard
            $stats = [
                'total' => Product::count(),
                'active' => Product::where('status', 'active')->count(),
                'with_expiry' => Product::where('has_expiry', true)->count(),
                'batch_tracked' => Product::where('track_batches', true)->count(),
                'low_stock' => Product::whereColumn('stock', '<=', 'minimum_stock')
                    ->where('stock', '>', 0)->count(),
                'out_of_stock' => Product::where('stock', '<=', 0)->count(),
                'total_stock_value' => Product::sum(DB::raw('cost_price * stock')),
                'taxable_products' => Product::where('has_vat', true)->count(),
                'tax_exempt_products' => Product::where('has_vat', false)->count(),
            ];

            $categories = Category::all();
            $brands = Brand::all();
            $suppliers = Supplier::where('status', 'active')->get();
            $taxes = Tax::where('is_active', true)->get();

            return view('products.index', compact('products', 'stats', 'categories', 'brands', 'suppliers', 'taxes'));
            
        } catch (\Exception $e) {
            Log::error('Product index error: ' . $e->getMessage());
            return redirect()->route('dashboard')
                ->with('error', 'Failed to load products. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::all();
        $brands = Brand::all();
        $suppliers = Supplier::where('status', 'active')->get();
        $units = Unit::all();
        $taxes = Tax::where('is_active', true)->get();

        // Get default tax for pre-selection
        $defaultTax = Tax::where('is_default', true)->where('is_active', true)->first();

        return view('products.create', compact('categories', 'brands', 'suppliers', 'units', 'taxes', 'defaultTax'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            // Get validation rules from model
            $validationRules = Product::getValidationRules();
            
            // Add additional rules for form-specific fields
            $validationRules['barcode'] = 'nullable|string|max:100|unique:products,barcode';
            $validationRules['ref_number'] = 'nullable|string|max:100|unique:products,ref_number';
            
            // Validate tax selection if has_vat is true
            $validationRules['tax_id'] = 'required_if:has_vat,1|nullable|exists:taxes,id';
            
            // Get custom validation messages
            $validationMessages = Product::getValidationMessages();
            $validationMessages['tax_id.required_if'] = 'Please select a tax rate for taxable products.';
            
            // Validate the request with custom messages
            $validated = $request->validate($validationRules, $validationMessages);

            // Add default values for checkbox fields that might be missing
            $checkboxFields = [
                'has_vat',
                'has_expiry',
                'track_batches',
                'is_featured',
                'is_new',
                'is_bestseller',
                'is_on_sale'
            ];
            
            foreach ($checkboxFields as $field) {
                if (!isset($validated[$field])) {
                    $validated[$field] = false;
                }
            }

            // Prepare product data
            $productData = [
                // Basic Information
                'sku' => $validated['sku'],
                'barcode' => $validated['barcode'] ?? null,
                'name' => $validated['name'],
                'slug' => \Illuminate\Support\Str::slug($validated['name']),
                
                // Descriptions
                'description' => $validated['description'] ?? null,
                'short_description' => $validated['short_description'] ?? null,
                
                // Relationships
                'category_id' => $validated['category_id'] ?? null,
                'brand_id' => $validated['brand_id'] ?? null,
                'supplier_id' => $validated['supplier_id'] ?? null,
                'unit_id' => $validated['unit_id'] ?? null,
                'tax_id' => $validated['has_vat'] ? ($validated['tax_id'] ?? null) : null,
                
                // Reference Numbers
                'ref_number' => $validated['ref_number'] ?? null,
                
                // Inventory
                'stock' => $validated['stock'],
                'reorder_point' => $validated['reorder_point'],
                'minimum_stock' => $validated['minimum_stock'],
                'maximum_stock' => $validated['maximum_stock'] ?? null,
                
                // Pricing
                'cost_price' => $validated['cost_price'],
                'sale_price' => $validated['sale_price'],
                'wholesale_price' => $validated['wholesale_price'] ?? null,
                'discount_price' => $validated['discount_price'] ?? null,
                'discount_percent' => $validated['discount_percent'] ?? null,
                'has_vat' => $validated['has_vat'] ?? false, // Changed to false by default
                
                // Expiry & Batch Tracking
                'has_expiry' => $validated['has_expiry'] ?? false,
                'track_batches' => $validated['track_batches'] ?? false,
                'expiry_warning_days' => $validated['has_expiry'] ? ($validated['expiry_warning_days'] ?? 30) : 0,
                'expiry_alert_level' => $validated['has_expiry'] ? 'warning' : 'normal',
                'manage_batch_numbers' => $validated['track_batches'] ?? false,
                
                // Default settings
                'track_inventory' => true,
                'allow_backorders' => false,
                'manage_serial_numbers' => false,
                
                // Status & Flags
                'status' => $validated['status'],
                'is_featured' => $validated['is_featured'] ?? false,
                'is_new' => $validated['is_new'] ?? false,
                'is_bestseller' => $validated['is_bestseller'] ?? false,
                'is_on_sale' => ($validated['discount_price'] ?? 0) > 0 || ($validated['discount_percent'] ?? 0) > 0,
                
                // Calculated fields
                'stock_status' => $validated['stock'] > 0 ? 'in_stock' : 'out_of_stock',
            ];

            // Auto-enable batch tracking if expiry is enabled
            if ($productData['has_expiry'] && !$productData['track_batches']) {
                $productData['track_batches'] = true;
                $productData['manage_batch_numbers'] = true;
            }

            // Calculate final price
            $productData['final_price'] = $this->calculateFinalPrice($productData);

            // Create the product
            $product = Product::create($productData);

            // Handle initial stock if provided
            if ($productData['stock'] > 0) {
                if ($productData['track_batches']) {
                    // Create initial batch for batch-tracked products
                    $batch = ProductBatch::create([
                        'product_id' => $product->id,
                        'batch_number' => 'INITIAL-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 6)),
                        'initial_quantity' => $productData['stock'],
                        'current_quantity' => $productData['stock'],
                        'batch_cost_price' => $productData['cost_price'],
                        'batch_sale_price' => $productData['sale_price'],
                        'created_by' => auth()->id(),
                    ]);

                    // Log batch movement
                    BatchStockMovement::create([
                        'batch_id' => $batch->id,
                        'product_id' => $product->id,
                        'movement_type' => 'in',
                        'quantity' => $productData['stock'],
                        'quantity_before' => 0,
                        'reason' => 'initial_stock',
                        'reference_number' => 'INIT-' . now()->format('YmdHis'),
                        'notes' => 'Initial product stock',
                        'user_id' => auth()->id(),
                    ]);
                } else {
                    // Log general stock movement
                    StockMovement::create([
                        'product_id' => $product->id,
                        'type' => 'in',
                        'qty_change' => $productData['stock'],
                        'qty_before' => 0,
                        'reason' => 'initial_stock',
                        'notes' => 'Initial product stock',
                        'user_id' => auth()->id(),
                    ]);
                }
            }

            // Record initial price history
            PriceHistory::create([
                'product_id' => $product->id,
                'user_id' => auth()->id(),
                'price_type' => 'cost_price',
                'old_price' => 0,
                'new_price' => $product->cost_price,
                'change_reason' => 'initial_setup',
                'notes' => 'Initial product setup',
                'effective_from' => now(),
            ]);

            PriceHistory::create([
                'product_id' => $product->id,
                'user_id' => auth()->id(),
                'price_type' => 'sale_price',
                'old_price' => 0,
                'new_price' => $product->sale_price,
                'change_reason' => 'initial_setup',
                'notes' => 'Initial product setup',
                'effective_from' => now(),
            ]);

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties($productData)
                ->log('created product');

            DB::commit();

            // Get tax info for success message
            $taxInfo = '';
            if ($product->has_vat && $product->tax) {
                $taxInfo = "Tax: {$product->tax->name} ({$product->tax->rate}%)";
            } else {
                $taxInfo = 'Product is tax-exempt';
            }

            return redirect()->route('products.show', $product)
                ->with('success', "Product created successfully! {$taxInfo}")
                ->with('product_id', $product->id);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            Log::error('Product validation error: ' . $e->getMessage());
            
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput()
                ->with('error', 'Please fix the validation errors.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product store error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->with('error', 'Failed to create product: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
/**
 * Display the specified resource.
 */
public function show(Product $product)
{
    try {
        // Load relationships
        $product->load([
            'category', 
            'brand', 
            'supplier', 
            'unit', 
            'tax',
            'batches' => function($query) {
                $query->orderBy('expiry_date')->orderBy('created_at');
            },
            'priceHistories' => function($query) {
                $query->latest()->limit(10);
            }
        ]);

        // Calculate expiry summary
        $expirySummary = [
            'has_expiry' => $product->has_expiry,
            'alert_level' => 'normal',
            'expired_quantity' => 0,
            'expiring_soon_quantity' => 0,
            'valid_quantity' => $product->stock,
        ];

        if ($product->has_expiry && $product->batches->isNotEmpty()) {
            $expiredBatches = $product->batches->where('expiry_date', '<', now());
            $expiringSoonBatches = $product->batches->whereBetween('expiry_date', [now(), now()->addDays($product->expiry_warning_days ?? 30)]);
            
            $expirySummary['expired_quantity'] = $expiredBatches->sum('current_quantity');
            $expirySummary['expiring_soon_quantity'] = $expiringSoonBatches->sum('current_quantity');
            $expirySummary['valid_quantity'] = $product->stock - $expirySummary['expired_quantity'] - $expirySummary['expiring_soon_quantity'];
            
            if ($expirySummary['expired_quantity'] > 0) {
                $expirySummary['alert_level'] = 'critical';
            } elseif ($expirySummary['expiring_soon_quantity'] > 0) {
                $expirySummary['alert_level'] = 'warning';
            }
        }

        // Get tax information
        $taxInfo = [
            'has_vat' => $product->has_vat,
            'is_vatable' => $product->is_vatable,
            'tax_rate' => $product->tax_rate,
            'tax_name' => $product->tax ? $product->tax->name : 'No Tax',
            'tax_code' => $product->tax ? $product->tax->code : 'NONE',
        ];


// Calculate tax examples - FIXED VERSION

    $taxExamples = [
        'single_unit' => [
            'price_excl_tax' => $product->final_price,
            'tax_amount' => 0,
            'price_with_tax' => $product->final_price,
            'tax_rate' => $product->tax ? $product->tax->rate : 0
        ],
        'ten_units' => [
            'price_excl_tax' => $product->final_price * 10,
            'tax_amount' => 0,
            'price_with_tax' => $product->final_price * 10,
            'tax_rate' => $product->tax ? $product->tax->rate : 0
        ]
    ];

    // Calculate actual tax if product is vatable
    if ($product->is_vatable && method_exists($product, 'calculateTaxDetails')) {
        try {
            $singleUnit = $product->calculateTaxDetails($product->final_price, 1);
            $tenUnits = $product->calculateTaxDetails($product->final_price, 10);
            
            $taxExamples = [
                'single_unit' => [
                    'price_excl_tax' => $singleUnit['price_without_tax'],
                    'tax_amount' => $singleUnit['tax_amount'],
                    'price_with_tax' => $singleUnit['price_with_tax'],
                    'tax_rate' => $singleUnit['tax_rate']
                ],
                'ten_units' => [
                    'price_excl_tax' => $tenUnits['price_without_tax'],
                    'tax_amount' => $tenUnits['tax_amount'],
                    'price_with_tax' => $tenUnits['price_with_tax'],
                    'tax_rate' => $tenUnits['tax_rate']
                ]
            ];
        } catch (\Exception $e) {
            Log::warning('Tax calculation error: ' . $e->getMessage());
        }
    }



        // Override with actual tax calculations if product is vatable
        if ($product->is_vatable) {
            // Check if calculateTaxDetails method exists
            if (method_exists($product, 'calculateTaxDetails')) {
                try {
                    $singleUnit = $product->calculateTaxDetails($product->final_price, 1);
                    $tenUnits = $product->calculateTaxDetails($product->final_price, 10);
                    
                    // Ensure we have all required keys
                    $taxExamples = [
                        'single_unit' => array_merge($taxExamples['single_unit'], $singleUnit),
                        'ten_units' => array_merge($taxExamples['ten_units'], $tenUnits),
                    ];
                } catch (\Exception $e) {
                    Log::warning('Tax calculation error for product ' . $product->id . ': ' . $e->getMessage());
                    // Keep default values if calculation fails
                }
            }
        }

        // Get batch statistics
        $batchStats = [
            'total_batches' => $product->batches()->count(),
            'active_batches' => $product->batches()->where('current_quantity', '>', 0)->count(),
            'expired_batches' => $product->batches()
                ->where('expiry_date', '<', now())
                ->where('current_quantity', '>', 0)->count(),
            'expiring_soon' => $product->batches()
                ->where('expiry_date', '>=', now())
                ->where('expiry_date', '<=', now()->addDays($product->expiry_warning_days ?? 30))
                ->where('current_quantity', '>', 0)->count(),
            'total_quantity' => $product->batches()->sum('current_quantity'),
            'total_value' => $product->batches()->sum(DB::raw('batch_cost_price * current_quantity')),
        ];

        // Get recent stock movements
        $stockMovements = BatchStockMovement::where('product_id', $product->id)
            ->with(['batch', 'user'])
            ->latest()
            ->limit(20)
            ->get();

        return view('products.show', compact(
            'product', 
            'batchStats', 
            'stockMovements', 
            'expirySummary',
            'taxInfo',
            'taxExamples'
        ));
        
    } catch (\Exception $e) {
        Log::error('Product show error: ' . $e->getMessage());
        return redirect()->route('products.index')
            ->with('error', 'Failed to load product details.');
    }
}
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = Category::all();
        $brands = Brand::all();
        $suppliers = Supplier::where('status', 'active')->get();
        $units = Unit::all();
        $taxes = Tax::where('is_active', true)->get();

        // Get default tax for selection
        $defaultTax = Tax::where('is_default', true)->where('is_active', true)->first();

        return view('products.edit', compact('product', 'categories', 'brands', 'suppliers', 'units', 'taxes', 'defaultTax'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            // Get validation rules for update (with product ID for unique rules)
            $validationRules = Product::getValidationRules($product->id);
            $validationRules['tax_id'] = 'required_if:has_vat,1|nullable|exists:taxes,id';
            
            $validationMessages = Product::getValidationMessages();
            $validationMessages['tax_id.required_if'] = 'Please select a tax rate for taxable products.';
            
            // Validate the request
            $validated = $request->validate($validationRules, $validationMessages);

            $oldData = $product->toArray();

            // Update slug if name changed
            if ($product->name !== $validated['name']) {
                $validated['slug'] = \Illuminate\Support\Str::slug($validated['name']);
            }

            // Clear tax_id if has_vat is false
            if (!$validated['has_vat']) {
                $validated['tax_id'] = null;
            }

            // Calculate final price
            $validated['final_price'] = $this->calculateFinalPrice($validated);

            // Record price history if prices changed
            if ($product->cost_price != $validated['cost_price']) {
                PriceHistory::create([
                    'product_id' => $product->id,
                    'user_id' => auth()->id(),
                    'price_type' => 'cost_price',
                    'old_price' => $product->cost_price,
                    'new_price' => $validated['cost_price'],
                    'change_reason' => 'price_update',
                    'notes' => 'Cost price updated',
                    'effective_from' => now(),
                ]);
            }

            if ($product->sale_price != $validated['sale_price']) {
                PriceHistory::create([
                    'product_id' => $product->id,
                    'user_id' => auth()->id(),
                    'price_type' => 'sale_price',
                    'old_price' => $product->sale_price,
                    'new_price' => $validated['sale_price'],
                    'change_reason' => 'price_update',
                    'notes' => 'Sale price updated',
                    'effective_from' => now(),
                ]);
            }

            // Update stock status if stock changed
            if ($product->stock != $validated['stock']) {
                $validated['stock_status'] = $validated['stock'] > 0 ? 'in_stock' : 'out_of_stock';
            }

            // Update the product
            $product->update($validated);

            // Log activity with changes
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'old' => $oldData,
                    'new' => $validated
                ])
                ->log('updated product');

            DB::commit();

            // Get tax info for success message
            $taxInfo = '';
            if ($product->has_vat && $product->tax) {
                $taxInfo = "Tax: {$product->tax->name} ({$product->tax->rate}%)";
            } else {
                $taxInfo = 'Product is tax-exempt';
            }

            return redirect()->route('products.show', $product)
                ->with('success', "Product updated successfully. {$taxInfo}");
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product update error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to update product: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        try {
            DB::beginTransaction();

            // Check if product has any stock
            if ($product->stock > 0) {
                return redirect()->route('products.index')
                    ->with('error', 'Cannot delete product with stock. Please clear stock first.');
            }

            // Check if product has any batches
            if ($product->batches()->exists()) {
                return redirect()->route('products.index')
                    ->with('error', 'Cannot delete product with batches. Please delete batches first.');
            }

            // Log activity before deletion
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->log('deleted product');

            $product->delete();

            DB::commit();

            return redirect()->route('products.index')
                ->with('success', 'Product deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product destroy error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to delete product. Please try again.');
        }
    }

    /**
     * Add stock to product.
     */
    public function addStock(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'quantity' => 'required|integer|min:1',
                'reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
                'batch_data' => 'nullable|array',
                'batch_data.batch_number' => 'nullable|string|max:100',
                'batch_data.expiry_date' => 'nullable|date|after:today',
                'batch_data.manufacture_date' => 'nullable|date|before_or_equal:today',
                'batch_data.cost_price' => 'nullable|numeric|min:0',
                'batch_data.sale_price' => 'nullable|numeric|min:0',
                'batch_data.supplier_id' => 'nullable|exists:suppliers,id',
                'batch_data.location' => 'nullable|string|max:100',
            ]);

            // Check if product has batch tracking
            if ($product->track_batches) {
                // Create or update batch
                $batch = ProductBatch::create([
                    'product_id' => $product->id,
                    'supplier_id' => $validated['batch_data']['supplier_id'] ?? null,
                    'batch_number' => $validated['batch_data']['batch_number'] ?? $this->generateBatchNumber(),
                    'expiry_date' => $validated['batch_data']['expiry_date'] ?? null,
                    'manufacture_date' => $validated['batch_data']['manufacture_date'] ?? now(),
                    'initial_quantity' => $validated['quantity'],
                    'current_quantity' => $validated['quantity'],
                    'batch_cost_price' => $validated['batch_data']['cost_price'] ?? $product->cost_price,
                    'batch_sale_price' => $validated['batch_data']['sale_price'] ?? $product->sale_price,
                    'warehouse_location' => $validated['batch_data']['location'] ?? null,
                    'created_by' => auth()->id(),
                ]);

                // Log batch movement
                BatchStockMovement::create([
                    'batch_id' => $batch->id,
                    'product_id' => $product->id,
                    'movement_type' => 'in',
                    'quantity' => $validated['quantity'],
                    'quantity_before' => 0,
                    'reason' => $validated['reason'],
                    'reference_number' => 'STOCK-ADD-' . now()->format('Ymd'),
                    'notes' => $validated['notes'] ?? null,
                    'user_id' => auth()->id(),
                ]);

                $message = "Added {$validated['quantity']} units to batch {$batch->batch_number}";
            } else {
                // Add to general stock
                $product->increment('stock', $validated['quantity']);
                $message = "Added {$validated['quantity']} units to general stock";
            }

            // Update product stock status
            $product->updateStockStatus();

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'quantity' => $validated['quantity'],
                    'reason' => $validated['reason'],
                    'batch_data' => $validated['batch_data'] ?? null,
                ])
                ->log('added stock to product');

            DB::commit();

            return redirect()->route('products.show', $product)
                ->with('success', 'Stock added successfully: ' . $message);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Add stock error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to add stock: ' . $e->getMessage());
        }
    }

    /**
     * Remove stock from product.
     */
    public function removeStock(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'quantity' => 'required|integer|min:1',
                'reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
                'method' => 'nullable|in:fifo,fefo',
                'reference_number' => 'nullable|string|max:100',
            ]);

            // Check if enough stock exists
            if ($product->stock < $validated['quantity']) {
                throw new \Exception("Insufficient stock. Available: {$product->stock}, Requested: {$validated['quantity']}");
            }

            if ($product->track_batches) {
                // Remove from batches using FIFO or FEFO
                $result = $this->removeFromBatches($product, $validated);
                $message = "Removed {$validated['quantity']} units from batches";
            } else {
                // Remove from general stock
                $product->decrement('stock', $validated['quantity']);
                $message = "Removed {$validated['quantity']} units from general stock";
            }

            // Update product stock status
            $product->updateStockStatus();

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'quantity' => $validated['quantity'],
                    'reason' => $validated['reason'],
                    'method' => $validated['method'] ?? 'fifo',
                ])
                ->log('removed stock from product');

            DB::commit();

            return redirect()->route('products.show', $product)
                ->with('success', 'Stock removed successfully: ' . $message);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Remove stock error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to remove stock: ' . $e->getMessage());
        }
    }

    /**
     * Show price history for a product.
     */
    public function priceHistory(Product $product)
    {
        try {
            $priceHistories = PriceHistory::where('product_id', $product->id)
                ->with('user')
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            return view('products.price-history', compact('product', 'priceHistories'));
            
        } catch (\Exception $e) {
            Log::error('Price history error: ' . $e->getMessage());
            return redirect()->route('products.show', $product)
                ->with('error', 'Failed to load price history.');
        }
    }

    /**
     * Show product batches.
     */
    public function batches(Product $product)
    {
        try {
            $batches = ProductBatch::where('product_id', $product->id)
                ->with(['supplier', 'creator'])
                ->orderBy('expiry_date')
                ->orderBy('created_at')
                ->paginate(20);

            $batchStats = [
                'total_batches' => $batches->total(),
                'active_batches' => $batches->where('current_quantity', '>', 0)->count(),
                'expired_batches' => $batches->where('expiry_date', '<', now())
                    ->where('current_quantity', '>', 0)->count(),
                'expiring_soon' => $batches->where('expiry_date', '>=', now())
                    ->where('expiry_date', '<=', now()->addDays($product->expiry_warning_days ?? 30))
                    ->where('current_quantity', '>', 0)->count(),
                'total_quantity' => $batches->sum('current_quantity'),
            ];

            return view('products.batches', compact('product', 'batches', 'batchStats'));
            
        } catch (\Exception $e) {
            Log::error('Product batches error: ' . $e->getMessage());
            return redirect()->route('products.show', $product)
                ->with('error', 'Failed to load batches.');
        }
    }

    /**
     * Show expiry report.
     */
    public function expiryReport(Request $request)
    {
        try {
            $query = ProductBatch::with(['product', 'supplier'])
                ->whereNotNull('expiry_date')
                ->where('current_quantity', '>', 0);

            // Filter by expiry range
            if ($request->has('days')) {
                $days = $request->input('days', 30);
                $query->where('expiry_date', '<=', now()->addDays($days));
            }

            // Filter by expired
            if ($request->has('expired')) {
                $query->where('expiry_date', '<', now());
            }

            $batches = $query->orderBy('expiry_date')
                ->paginate(20)
                ->withQueryString();

            $stats = [
                'total_expiring' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '<=', now()->addDays(30))
                    ->where('current_quantity', '>', 0)
                    ->count(),
                'expired' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '<', now())
                    ->where('current_quantity', '>', 0)
                    ->count(),
                'total_value' => ProductBatch::whereNotNull('expiry_date')
                    ->where('current_quantity', '>', 0)
                    ->sum(DB::raw('batch_cost_price * current_quantity')),
            ];

            return view('products.expiry-report', compact('batches', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('Expiry report error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to load expiry report.');
        }
    }

    /**
     * Set discount for expiring products.
     */
    public function setExpiryDiscount(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'discount_type' => 'required|in:amount,percent',
                'discount_value' => 'required|numeric|min:0',
                'discount_reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
            ]);

            // Calculate discount price
            if ($validated['discount_type'] === 'percent') {
                $discountPrice = $product->sale_price * (1 - ($validated['discount_value'] / 100));
                $discountPercent = $validated['discount_value'];
            } else {
                $discountPrice = $product->sale_price - $validated['discount_value'];
                $discountPercent = ($validated['discount_value'] / $product->sale_price) * 100;
            }

            // Ensure discount price is not negative
            if ($discountPrice < 0) {
                $discountPrice = 0;
                $discountPercent = 100;
            }

            // Update product with discount
            $product->update([
                'discount_price' => $discountPrice,
                'discount_percent' => $discountPercent,
                'is_on_sale' => true,
                'final_price' => $discountPrice,
            ]);

            // Log discount activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'original_price' => $product->sale_price,
                    'discount_price' => $discountPrice,
                    'discount_percent' => $discountPercent,
                    'discount_reason' => $validated['discount_reason'],
                    'notes' => $validated['notes'] ?? null,
                ])
                ->log('set expiry discount');

            DB::commit();

            return redirect()->route('products.show', $product)
                ->with('success', 'Discount set successfully for expiring product.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Set expiry discount error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to set discount: ' . $e->getMessage());
        }
    }

    /**
     * Bulk update products.
     */
    public function bulkUpdate(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_ids' => 'required|array',
                'product_ids.*' => 'exists:products,id',
                'data' => 'required|array',
                'data.status' => 'nullable|in:active,inactive',
                'data.category_id' => 'nullable|exists:categories,id',
                'data.brand_id' => 'nullable|exists:brands,id',
                'data.supplier_id' => 'nullable|exists:suppliers,id',
                'data.tax_id' => 'nullable|exists:taxes,id',
                'data.has_vat' => 'nullable|boolean',
            ]);

            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $updatedCount = 0;

            foreach ($products as $product) {
                // Clear tax_id if has_vat is being set to false
                if (isset($validated['data']['has_vat']) && !$validated['data']['has_vat']) {
                    $validated['data']['tax_id'] = null;
                }
                
                $product->update(array_filter($validated['data']));
                $updatedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Updated {$updatedCount} products successfully.",
                'updated_count' => $updatedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk update error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update products: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete products.
     */
    public function bulkDelete(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_ids' => 'required|array',
                'product_ids.*' => 'exists:products,id',
            ]);

            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $deletedCount = 0;

            foreach ($products as $product) {
                // Check if product can be deleted
                if ($product->stock > 0 || $product->batches()->exists()) {
                    continue;
                }

                // Log activity before deletion
                activity()
                    ->causedBy(auth()->user())
                    ->performedOn($product)
                    ->log('bulk deleted product');

                $product->delete();
                $deletedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Deleted {$deletedCount} products successfully.",
                'deleted_count' => $deletedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete products: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export products.
     */
    public function export(Request $request)
    {
        try {
            $query = Product::with(['category', 'brand', 'supplier', 'tax']);

            // Apply filters from request
            if ($request->has('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            $products = $query->get();

            $fileName = 'products-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];

            $callback = function() use ($products) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'ID', 'SKU', 'Name', 'Category', 'Brand', 'Supplier',
                    'Stock', 'Cost Price', 'Sale Price', 'Final Price', 
                    'Has VAT', 'Tax Rate', 'Tax Name', 'Status', 'Created At'
                ]);
                
                // Data
                foreach ($products as $product) {
                    fputcsv($file, [
                        $product->id,
                        $product->sku,
                        $product->name,
                        $product->category->name ?? '',
                        $product->brand->name ?? '',
                        $product->supplier->name ?? '',
                        $product->stock,
                        $product->cost_price,
                        $product->sale_price,
                        $product->final_price,
                        $product->has_vat ? 'Yes' : 'No',
                        $product->tax ? $product->tax->rate . '%' : '0%',
                        $product->tax ? $product->tax->name : 'No Tax',
                        $product->status,
                        $product->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Product export error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to export products.');
        }
    }

    /**
     * Search products for AJAX requests.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q');
            $limit = $request->input('limit', 10);
            $checkSku = $request->input('check_sku', false);

            if ($checkSku) {
                // Check if SKU exists
                $exists = Product::where('sku', $search)->exists();
                return response()->json(['exists' => $exists]);
            }

            $products = Product::where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('sku', 'like', "%{$search}%")
                          ->orWhere('barcode', 'like', "%{$search}%");
                })
                ->where('status', 'active')
                ->select('id', 'sku', 'name', 'barcode', 'stock', 'sale_price', 'discount_price', 'final_price', 'has_vat', 'tax_id')
                ->with('tax')
                ->limit($limit)
                ->get();

            return response()->json($products);
            
        } catch (\Exception $e) {
            Log::error('Product search error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /**
     * Show product statistics.
     */
    public function statistics()
    {
        try {
            // Product statistics
            $stats = [
                'total_products' => Product::count(),
                'active_products' => Product::where('status', 'active')->count(),
                'inactive_products' => Product::where('status', 'inactive')->count(),
                'featured_products' => Product::where('is_featured', true)->count(),
                'products_with_expiry' => Product::where('has_expiry', true)->count(),
                'batch_tracked_products' => Product::where('track_batches', true)->count(),
                'taxable_products' => Product::where('has_vat', true)->count(),
                'tax_exempt_products' => Product::where('has_vat', false)->count(),
                'total_stock_value' => Product::sum(DB::raw('cost_price * stock')),
                'total_sale_value' => Product::sum(DB::raw('final_price * stock')),
                'avg_cost_price' => Product::avg('cost_price'),
                'avg_sale_price' => Product::avg('sale_price'),
                'avg_profit_margin' => Product::avg(DB::raw('((sale_price - cost_price) / cost_price) * 100')),
            ];

            // Category distribution
            $categoryDistribution = Category::withCount('products')
                ->orderByDesc('products_count')
                ->limit(10)
                ->get();

            // Brand distribution
            $brandDistribution = Brand::withCount('products')
                ->orderByDesc('products_count')
                ->limit(10)
                ->get();

            // Stock status distribution
            $stockStatus = [
                'in_stock' => Product::where('stock', '>', 0)->count(),
                'low_stock' => Product::whereColumn('stock', '<=', 'minimum_stock')
                    ->where('stock', '>', 0)->count(),
                'out_of_stock' => Product::where('stock', '<=', 0)->count(),
            ];

            // Recent products
            $recentProducts = Product::with(['category', 'brand'])
                ->latest()
                ->limit(10)
                ->get();

            return view('products.statistics', compact(
                'stats', 
                'categoryDistribution', 
                'brandDistribution', 
                'stockStatus', 
                'recentProducts'
            ));
            
        } catch (\Exception $e) {
            Log::error('Product statistics error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to load product statistics.');
        }
    }

    /**
     * Import products view.
     */
    public function import()
    {
        $taxes = Tax::where('is_active', true)->get();
        $defaultTax = Tax::where('is_default', true)->where('is_active', true)->first();
        
        return view('products.import', compact('taxes', 'defaultTax'));
    }

    /**
     * Process product import.
     */
    public function processImport(Request $request)
    {
        try {
            $request->validate([
                'file' => 'required|mimes:csv,txt|max:2048',
                'default_tax_id' => 'nullable|exists:taxes,id',
            ]);

            // Process CSV file
            $file = $request->file('file');
            $csvData = array_map('str_getcsv', file($file->getPathname()));
            
            // Remove header
            $header = array_shift($csvData);
            
            $imported = 0;
            $skipped = 0;
            $errors = [];

            DB::beginTransaction();

            // Get default tax from request or system default
            $defaultTaxId = $request->input('default_tax_id');
            if (!$defaultTaxId) {
                $defaultTax = Tax::where('is_default', true)->where('is_active', true)->first();
                $defaultTaxId = $defaultTax ? $defaultTax->id : null;
            }

            foreach ($csvData as $row) {
                try {
                    // Map CSV columns to product fields
                    $productData = array_combine($header, $row);
                    
                    // Validate required fields
                    if (empty($productData['sku']) || empty($productData['name'])) {
                        $skipped++;
                        continue;
                    }

                    // Check if SKU already exists
                    $existingProduct = Product::where('sku', $productData['sku'])->first();
                    if ($existingProduct) {
                        $skipped++;
                        continue;
                    }

                    // Determine tax settings
                    $hasVat = isset($productData['has_vat']) ? filter_var($productData['has_vat'], FILTER_VALIDATE_BOOLEAN) : false;
                    $taxId = null;
                    
                    if ($hasVat) {
                        // Try to find tax by name or rate
                        if (isset($productData['tax_name'])) {
                            $tax = Tax::where('name', $productData['tax_name'])->first();
                            if ($tax) {
                                $taxId = $tax->id;
                            }
                        } elseif (isset($productData['tax_rate'])) {
                            $tax = Tax::where('rate', $productData['tax_rate'])->first();
                            if ($tax) {
                                $taxId = $tax->id;
                            }
                        }
                        
                        // Fallback to default tax
                        if (!$taxId && $defaultTaxId) {
                            $taxId = $defaultTaxId;
                        }
                    }

                    // Create product
                    $product = Product::create([
                        'sku' => $productData['sku'],
                        'name' => $productData['name'],
                        'category_id' => $this->findOrCreateCategory($productData['category'] ?? null),
                        'brand_id' => $this->findOrCreateBrand($productData['brand'] ?? null),
                        'description' => $productData['description'] ?? null,
                        'short_description' => $productData['short_description'] ?? null,
                        'stock' => intval($productData['stock'] ?? 0),
                        'cost_price' => floatval($productData['cost_price'] ?? 0),
                        'sale_price' => floatval($productData['sale_price'] ?? 0),
                        'has_vat' => $hasVat,
                        'tax_id' => $taxId,
                        'final_price' => floatval($productData['sale_price'] ?? 0),
                        'status' => 'active',
                        'slug' => \Illuminate\Support\Str::slug($productData['name']),
                    ]);

                    $imported++;
                    
                } catch (\Exception $e) {
                    $errors[] = "Row " . ($imported + $skipped + 1) . ": " . $e->getMessage();
                    $skipped++;
                }
            }

            DB::commit();

            return redirect()->route('products.index')
                ->with('success', "Imported {$imported} products. Skipped {$skipped} rows.")
                ->with('import_errors', $errors);
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product import error: ' . $e->getMessage());
            return redirect()->route('products.import')
                ->with('error', 'Failed to import products: ' . $e->getMessage());
        }
    }

    /**
     * Pricing management index.
     */
    public function pricingIndex(Request $request)
    {
        try {
            $query = Product::with(['category', 'brand', 'tax']);

            // Search functionality
            if ($request->has('search')) {
                $search = $request->input('search');
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%");
                });
            }

            // Filter by category
            if ($request->has('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            // Filter by brand
            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->brand_id);
            }

            // Filter by tax status
            if ($request->has('has_vat') && $request->has_vat != '') {
                $query->where('has_vat', $request->has_vat == '1');
            }

            // Filter by margin range
            if ($request->has('margin_range')) {
                switch ($request->margin_range) {
                    case 'high':
                        $query->whereRaw('((sale_price - cost_price) / sale_price) * 100 > 40');
                        break;
                    case 'medium':
                        $query->whereRaw('((sale_price - cost_price) / sale_price) * 100 BETWEEN 20 AND 40');
                        break;
                    case 'low':
                        $query->whereRaw('((sale_price - cost_price) / sale_price) * 100 < 20')
                              ->whereRaw('((sale_price - cost_price) / sale_price) * 100 > 0');
                        break;
                    case 'negative':
                        $query->whereRaw('((sale_price - cost_price) / sale_price) * 100 <= 0');
                        break;
                }
            }

            $products = $query->orderBy('name')->paginate(20)->withQueryString();

            $categories = Category::all();
            $brands = Brand::all();

            return view('products.pricing.index', compact('products', 'categories', 'brands'));
            
        } catch (\Exception $e) {
            Log::error('Pricing index error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to load pricing management.');
        }
    }

    /**
     * Bulk pricing update.
     */
    public function bulkPricingUpdate(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_ids' => 'required|array',
                'product_ids.*' => 'exists:products,id',
                'price_type' => 'required|in:sale_price,cost_price,wholesale_price',
                'update_type' => 'required|in:set,increase,decrease,percentage',
                'value' => 'required|numeric',
                'change_reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
            ]);

            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $updatedCount = 0;

            foreach ($products as $product) {
                $oldPrice = $product->{$validated['price_type']};
                $newPrice = $oldPrice;

                switch ($validated['update_type']) {
                    case 'set':
                        $newPrice = $validated['value'];
                        break;
                    case 'increase':
                        $newPrice = $oldPrice + $validated['value'];
                        break;
                    case 'decrease':
                        $newPrice = $oldPrice - $validated['value'];
                        break;
                    case 'percentage':
                        $newPrice = $oldPrice * (1 + ($validated['value'] / 100));
                        break;
                }

                // Ensure price is not negative
                if ($newPrice < 0) {
                    $newPrice = 0;
                }

                // Update the price
                $product->update([
                    $validated['price_type'] => $newPrice,
                ]);

                // Recalculate final price
                $product->final_price = $this->calculateFinalPrice($product->toArray());
                $product->save();

                // Record price history
                PriceHistory::create([
                    'product_id' => $product->id,
                    'user_id' => auth()->id(),
                    'price_type' => $validated['price_type'],
                    'old_price' => $oldPrice,
                    'new_price' => $newPrice,
                    'change_reason' => $validated['change_reason'],
                    'notes' => $validated['notes'] ?? null,
                    'effective_from' => now(),
                ]);

                $updatedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Prices updated for {$updatedCount} products.",
                'updated_count' => $updatedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk pricing update error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update prices: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update individual product price.
     */

    public function updateSinglePrice(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'price_type' => 'required|in:sale_price,cost_price,wholesale_price',
                'new_price' => 'required|numeric|min:0',
                'change_reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
            ]);

            $oldPrice = $product->{$validated['price_type']};
            $newPrice = $validated['new_price'];

            // Update the price
            $product->update([
                $validated['price_type'] => $newPrice,
            ]);

            // Recalculate final price
            $product->final_price = $this->calculateFinalPrice($product->toArray());
            $product->save();

            // Record price history
            PriceHistory::create([
                'product_id' => $product->id,
                'user_id' => auth()->id(),
                'price_type' => $validated['price_type'],
                'old_price' => $oldPrice,
                'new_price' => $newPrice,
                'change_reason' => $validated['change_reason'],
                'notes' => $validated['notes'] ?? null,
                'effective_from' => now(),
            ]);

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'price_type' => $validated['price_type'],
                    'old_price' => $oldPrice,
                    'new_price' => $newPrice,
                    'reason' => $validated['change_reason'],
                ])
                ->log('updated product price');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Price updated successfully.',
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update price error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update price: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set discount for product.
     */
    public function setDiscount(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'discount_type' => 'required|in:amount,percent',
                'discount_value' => 'required|numeric|min:0',
                'reason' => 'nullable|string|max:255',
            ]);

            // Calculate discount price
            if ($validated['discount_type'] === 'percent') {
                $discountPrice = $product->sale_price * (1 - ($validated['discount_value'] / 100));
                $discountPercent = $validated['discount_value'];
            } else {
                $discountPrice = $product->sale_price - $validated['discount_value'];
                $discountPercent = ($validated['discount_value'] / $product->sale_price) * 100;
            }

            // Ensure discount price is not negative
            if ($discountPrice < 0) {
                $discountPrice = 0;
                $discountPercent = 100;
            }

            // Update product with discount
            $product->update([
                'discount_price' => $discountPrice,
                'discount_percent' => $discountPercent,
                'final_price' => $discountPrice,
                'is_on_sale' => true,
            ]);

            // Log discount activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->withProperties([
                    'original_price' => $product->sale_price,
                    'discount_price' => $discountPrice,
                    'discount_percent' => $discountPercent,
                    'reason' => $validated['reason'] ?? null,
                ])
                ->log('set product discount');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Discount applied successfully.',
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Set discount error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to apply discount: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Clear discount for product.
     */
    public function clearDiscount(Request $request, Product $product)
    {
        try {
            DB::beginTransaction();

            // Remove discount
            $product->update([
                'discount_price' => null,
                'discount_percent' => null,
                'final_price' => $product->sale_price,
                'is_on_sale' => false,
            ]);

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($product)
                ->log('cleared product discount');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Discount cleared successfully.',
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Clear discount error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear discount: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show stock movements for a product with charts.
     */
    public function stockMovements(Product $product)
    {
        try {
            // Load relationships
            $product->load(['category', 'brand', 'supplier']);
            
            // Get all stock movements with pagination
            $stockMovements = BatchStockMovement::where('product_id', $product->id)
                ->with(['batch', 'user'])
                ->latest()
                ->paginate(20);
            
            // Get statistics with net_movement calculation
            $totalIn = BatchStockMovement::where('product_id', $product->id)
                ->where('movement_type', 'in')
                ->sum('quantity') ?? 0;
                
            $totalOut = abs(BatchStockMovement::where('product_id', $product->id)
                ->where('movement_type', 'out')
                ->sum('quantity')) ?? 0;
            
            $stats = [
                'total_movements' => $stockMovements->total(),
                'total_in' => $totalIn,
                'total_out' => $totalOut,
                'net_movement' => $totalIn - $totalOut,
                'current_stock' => $product->stock,
            ];
            
            // Get chart data for last 30 days
            $chartData = $this->getStockMovementChartData($product);
            
            return view('products.stock-movements', compact('product', 'stockMovements', 'stats', 'chartData'));
            
        } catch (\Exception $e) {
            Log::error('Stock movements error: ' . $e->getMessage());
            return redirect()->route('products.show', $product)
                ->with('error', 'Failed to load stock movements.');
        }
    }

    /**
     * Get stock movement chart data.
     */
    private function getStockMovementChartData(Product $product)
    {
        $days = 30; // Last 30 days
        $endDate = now();
        $startDate = now()->subDays($days);
        
        // Initialize arrays
        $labels = [];
        $inData = [];
        $outData = [];
        $netData = [];
        
        // Generate labels for each day
        for ($i = $days; $i >= 0; $i--) {
            $date = $startDate->copy()->addDays($i);
            $labels[] = $date->format('M d');
            
            // Get movements for this day
            $dayMovements = BatchStockMovement::where('product_id', $product->id)
                ->whereDate('created_at', $date->format('Y-m-d'))
                ->get();
            
            $dayIn = $dayMovements->where('movement_type', 'in')->sum('quantity');
            $dayOut = abs($dayMovements->where('movement_type', 'out')->sum('quantity'));
            $dayNet = $dayIn - $dayOut;
            
            $inData[] = $dayIn;
            $outData[] = $dayOut;
            $netData[] = $dayNet;
        }
        
        // Get movement type distribution
        $typeDistribution = BatchStockMovement::where('product_id', $product->id)
            ->select('movement_type', DB::raw('SUM(ABS(quantity)) as total_quantity'))
            ->groupBy('movement_type')
            ->get()
            ->pluck('total_quantity', 'movement_type');
        
        // Get reason distribution
        $reasonDistribution = BatchStockMovement::where('product_id', $product->id)
            ->select('reason', DB::raw('SUM(ABS(quantity)) as total_quantity'))
            ->groupBy('reason')
            ->orderByDesc('total_quantity')
            ->limit(10)
            ->get();
        
        // Get top 5 batches by movement volume
        $batchDistribution = BatchStockMovement::where('product_id', $product->id)
            ->whereNotNull('batch_id')
            ->with('batch')
            ->select('batch_id', DB::raw('SUM(ABS(quantity)) as total_quantity'))
            ->groupBy('batch_id')
            ->orderByDesc('total_quantity')
            ->limit(5)
            ->get()
            ->map(function($item) {
                return [
                    'batch_number' => $item->batch->batch_number ?? 'Unknown',
                    'quantity' => $item->total_quantity
                ];
            });
        
        return [
            'daily' => [
                'labels' => $labels,
                'in' => $inData,
                'out' => $outData,
                'net' => $netData,
            ],
            'typeDistribution' => [
                'in' => $typeDistribution['in'] ?? 0,
                'out' => $typeDistribution['out'] ?? 0,
            ],
            'reasonDistribution' => $reasonDistribution,
            'batchDistribution' => $batchDistribution,
        ];
    }

    /**
     * Get recent stock movements for product show page.
     */
    public function getRecentStockMovements(Product $product)
    {
        try {
            $stockMovements = BatchStockMovement::where('product_id', $product->id)
                ->with(['batch', 'user'])
                ->latest()
                ->limit(10)
                ->get();
            
            // Get summary for the last 7 days
            $weekSummary = $this->getWeekStockSummary($product);
            
            return response()->json([
                'success' => true,
                'movements' => $stockMovements,
                'week_summary' => $weekSummary,
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get recent stock movements error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load stock movements'
            ], 500);
        }
    }

    /**
     * Get weekly stock summary.
     */
    private function getWeekStockSummary(Product $product)
    {
        $startDate = now()->subDays(6);
        
        $movements = BatchStockMovement::where('product_id', $product->id)
            ->where('created_at', '>=', $startDate)
            ->get();
        
        $weekIn = $movements->where('movement_type', 'in')->sum('quantity');
        $weekOut = abs($movements->where('movement_type', 'out')->sum('quantity'));
        $weekNet = $weekIn - $weekOut;
        
        // Daily breakdown
        $dailyBreakdown = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $dayDate = $date->format('Y-m-d');
            
            $dayMovements = $movements->filter(function($movement) use ($dayDate) {
                return $movement->created_at->format('Y-m-d') === $dayDate;
            });
            
            $dayIn = $dayMovements->where('movement_type', 'in')->sum('quantity');
            $dayOut = abs($dayMovements->where('movement_type', 'out')->sum('quantity'));
            $dayNet = $dayIn - $dayOut;
            
            $dailyBreakdown[] = [
                'date' => $date->format('D'),
                'in' => $dayIn,
                'out' => $dayOut,
                'net' => $dayNet,
            ];
        }
        
        return [
            'week_in' => $weekIn,
            'week_out' => $weekOut,
            'week_net' => $weekNet,
            'daily_breakdown' => $dailyBreakdown,
        ];
    }

    /**
     * Export stock movements.
     */
    public function exportStockMovements(Product $product, Request $request)
    {
        try {
            $query = BatchStockMovement::where('product_id', $product->id)
                ->with(['batch', 'user']);
            
            // Apply filters if any
            if ($request->has('start_date')) {
                $query->where('created_at', '>=', $request->start_date);
            }
            
            if ($request->has('end_date')) {
                $query->where('created_at', '<=', $request->end_date);
            }
            
            if ($request->has('movement_type')) {
                $query->where('movement_type', $request->movement_type);
            }
            
            $movements = $query->latest()->get();
            
            $fileName = 'stock-movements-' . $product->sku . '-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($movements, $product) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Product', 'SKU', 'Date', 'Time', 'Movement Type', 'Quantity',
                    'Batch Number', 'Reason', 'Reference Number', 'Notes', 'User'
                ]);
                
                // Data
                foreach ($movements as $movement) {
                    fputcsv($file, [
                        $product->name,
                        $product->sku,
                        $movement->created_at->format('Y-m-d'),
                        $movement->created_at->format('H:i:s'),
                        $movement->movement_type == 'in' ? 'Stock In' : 'Stock Out',
                        $movement->movement_type == 'in' ? $movement->quantity : -$movement->quantity,
                        $movement->batch->batch_number ?? 'N/A',
                        $movement->reason,
                        $movement->reference_number ?? 'N/A',
                        $movement->notes ?? 'N/A',
                        $movement->user->name ?? 'System',
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Export stock movements error: ' . $e->getMessage());
            return redirect()->route('products.stock-movements', $product)
                ->with('error', 'Failed to export stock movements.');
        }
    }

    /**
     * Helper: Calculate final price.
     */
    private function calculateFinalPrice(array $data): float
    {
        $salePrice = $data['sale_price'];
        
        // Apply discount if exists
        if (!empty($data['discount_price']) && $data['discount_price'] > 0) {
            return $data['discount_price'];
        }
        
        if (!empty($data['discount_percent']) && $data['discount_percent'] > 0) {
            return $salePrice * (1 - ($data['discount_percent'] / 100));
        }
        
        return $salePrice;
    }

    /**
     * Helper: Generate batch number.
     */
    private function generateBatchNumber(): string
    {
        return 'BATCH-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 6));
    }

    /**
     * Helper: Remove stock from batches.
     */
    private function removeFromBatches(Product $product, array $data): array
    {
        $method = $data['method'] ?? 'fifo';
        $remaining = $data['quantity'];
        $allocations = [];
        
        // Get batches based on method
        $query = ProductBatch::where('product_id', $product->id)
            ->where('current_quantity', '>', 0);
            
        if ($method === 'fefo') {
            $query->orderBy('expiry_date')->orderBy('created_at');
        } else {
            $query->orderBy('created_at');
        }
        
        $batches = $query->get();
        
        foreach ($batches as $batch) {
            if ($remaining <= 0) break;
            
            $take = min($batch->current_quantity, $remaining);
            
            if ($take > 0) {
                $batch->decrement('current_quantity', $take);
                
                // Log batch movement
                BatchStockMovement::create([
                    'batch_id' => $batch->id,
                    'product_id' => $product->id,
                    'movement_type' => 'out',
                    'quantity' => -$take,
                    'quantity_before' => $batch->current_quantity + $take,
                    'reason' => $data['reason'],
                    'reference_number' => $data['reference_number'] ?? null,
                    'notes' => $data['notes'] ?? 'Stock reduction',
                    'user_id' => auth()->id(),
                ]);
                
                $allocations[] = [
                    'batch_id' => $batch->id,
                    'batch_number' => $batch->batch_number,
                    'expiry_date' => $batch->expiry_date?->format('Y-m-d'),
                    'quantity' => $take,
                ];
                
                $remaining -= $take;
            }
        }
        
        if ($remaining > 0) {
            throw new \Exception("Insufficient available stock in batches. Need {$remaining} more units.");
        }
        
        // Update product stock
        $product->decrement('stock', $data['quantity']);
        
        return [
            'success' => true,
            'allocations' => $allocations,
            'method' => $method,
        ];
    }

    /**
     * Helper: Find or create category.
     */
    private function findOrCreateCategory(?string $categoryName): ?int
    {
        if (!$categoryName) {
            return null;
        }

        $category = Category::where('name', $categoryName)->first();
        
        if (!$category) {
            $category = Category::create([
                'name' => $categoryName,
                'slug' => \Illuminate\Support\Str::slug($categoryName),
                'status' => 'active',
            ]);
        }

        return $category->id;
    }

    /**
     * Helper: Find or create brand.
     */
    private function findOrCreateBrand(?string $brandName): ?int
    {
        if (!$brandName) {
            return null;
        }

        $brand = Brand::where('name', $brandName)->first();
        
        if (!$brand) {
            $brand = Brand::create([
                'name' => $brandName,
            ]);
        }

        return $brand->id;
    }

    /**
     * Bulk set expiry discount for multiple products.
     */
    public function bulkSetExpiryDiscount(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_ids' => 'required|array',
                'product_ids.*' => 'exists:products,id',
                'discount_type' => 'required|in:amount,percent',
                'discount_value' => 'required|numeric|min:0',
                'discount_reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
            ]);

            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $updatedCount = 0;

            foreach ($products as $product) {
                // Calculate discount price
                if ($validated['discount_type'] === 'percent') {
                    $discountPrice = $product->sale_price * (1 - ($validated['discount_value'] / 100));
                    $discountPercent = $validated['discount_value'];
                } else {
                    $discountPrice = $product->sale_price - $validated['discount_value'];
                    $discountPercent = ($validated['discount_value'] / $product->sale_price) * 100;
                }

                // Ensure discount price is not negative
                if ($discountPrice < 0) {
                    $discountPrice = 0;
                    $discountPercent = 100;
                }

                // Update product with discount
                $product->update([
                    'discount_price' => $discountPrice,
                    'discount_percent' => $discountPercent,
                    'final_price' => $discountPrice,
                    'is_on_sale' => true,
                ]);

                // Log discount activity
                activity()
                    ->causedBy(auth()->user())
                    ->performedOn($product)
                    ->withProperties([
                        'original_price' => $product->sale_price,
                        'discount_price' => $discountPrice,
                        'discount_percent' => $discountPercent,
                        'discount_reason' => $validated['discount_reason'],
                        'notes' => $validated['notes'] ?? null,
                    ])
                    ->log('set expiry discount (bulk)');

                $updatedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Discount set for {$updatedCount} products.",
                'updated_count' => $updatedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk set expiry discount error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to set discount: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show inventory history with graphs.
     */
    public function inventoryHistory(Product $product)
    {
        try {
            // Load relationships
            $product->load(['category', 'brand', 'supplier']);
            
            // Get all inventory movements (both batch and regular)
            $movements = $this->getCombinedInventoryMovements($product);
            
            // Get detailed statistics
            $stats = $this->getInventoryStatistics($product);
            
            // Get chart data for last 90 days
            $chartData = $this->getInventoryChartData($product);
            
            // Get inventory trends
            $trends = $this->getInventoryTrends($product);
            
            // Get monthly summary
            $monthlySummary = $this->getMonthlyInventorySummary($product);
            
            return view('products.inventory-history', compact(
                'product', 
                'movements', 
                'stats', 
                'chartData', 
                'trends',
                'monthlySummary'
            ));
            
        } catch (\Exception $e) {
            Log::error('Inventory history error: ' . $e->getMessage());
            return redirect()->route('products.show', $product)
                ->with('error', 'Failed to load inventory history.');
        }
    }
    
    /**
     * Get combined inventory movements.
     */
    private function getCombinedInventoryMovements(Product $product)
    {
        // Get batch movements
        $batchMovements = BatchStockMovement::where('product_id', $product->id)
            ->with(['batch', 'user'])
            ->select([
                'id',
                'batch_id',
                'product_id',
                'movement_type',
                'quantity',
                'quantity_before',
                'reason',
                'reference_number',
                'notes',
                'user_id',
                'created_at',
                DB::raw('NULL as stock_movement_id'),
                DB::raw("'batch' as movement_source")
            ]);
        
        // Get regular stock movements
        $regularMovements = StockMovement::where('product_id', $product->id)
            ->with(['user'])
            ->select([
                'id',
                DB::raw('NULL as batch_id'),
                'product_id',
                DB::raw('UPPER(type) as movement_type'),
                'qty_change as quantity',
                'qty_before as quantity_before',
                'reason',
                DB::raw('NULL as reference_number'),
                'notes',
                'user_id',
                'created_at',
                DB::raw('id as stock_movement_id'),
                DB::raw("'regular' as movement_source")
            ]);
        
        // Combine and paginate
        return $batchMovements->union($regularMovements)
            ->orderBy('created_at', 'desc')
            ->paginate(30)
            ->through(function($item) {
                // Add dynamic relationships
                if ($item->movement_source === 'batch') {
                    $item->load(['batch', 'user']);
                } else {
                    $item->load(['user']);
                }
                return $item;
            });
    }
    
    /**
     * Get comprehensive inventory statistics.
     */
    private function getInventoryStatistics(Product $product)
    {
        // Batch movements statistics
        $batchStats = BatchStockMovement::where('product_id', $product->id)
            ->selectRaw('
                COUNT(*) as total_movements,
                SUM(CASE WHEN movement_type = "in" THEN quantity ELSE 0 END) as total_in,
                SUM(CASE WHEN movement_type = "out" THEN ABS(quantity) ELSE 0 END) as total_out,
                COUNT(DISTINCT DATE(created_at)) as active_days,
                COUNT(DISTINCT batch_id) as batches_involved
            ')->first();
        
        // Regular movements statistics
        $regularStats = StockMovement::where('product_id', $product->id)
            ->selectRaw('
                COUNT(*) as total_movements,
                SUM(CASE WHEN type = "in" THEN qty_change ELSE 0 END) as total_in,
                SUM(CASE WHEN type = "out" THEN ABS(qty_change) ELSE 0 END) as total_out
            ')->first();
        
        // Combined statistics
        $totalMovements = ($batchStats->total_movements ?? 0) + ($regularStats->total_movements ?? 0);
        $totalIn = ($batchStats->total_in ?? 0) + ($regularStats->total_in ?? 0);
        $totalOut = ($batchStats->total_out ?? 0) + ($regularStats->total_out ?? 0);
        
        // Get first and last movement dates
        $firstMovement = BatchStockMovement::where('product_id', $product->id)
            ->orderBy('created_at', 'asc')
            ->first();
        
        if (!$firstMovement) {
            $firstMovement = StockMovement::where('product_id', $product->id)
                ->orderBy('created_at', 'asc')
                ->first();
        }
        
        // Get daily average
        $daysActive = $batchStats->active_days ?? 0;
        $dailyAverage = $daysActive > 0 ? $totalMovements / $daysActive : 0;
        
        return [
            'total_movements' => $totalMovements,
            'total_in' => $totalIn,
            'total_out' => $totalOut,
            'net_movement' => $totalIn - $totalOut,
            'batches_involved' => $batchStats->batches_involved ?? 0,
            'active_days' => $daysActive,
            'daily_average' => round($dailyAverage, 2),
            'first_movement_date' => $firstMovement ? $firstMovement->created_at->format('M d, Y') : 'No movements',
            'current_stock' => $product->stock,
            'stock_value' => $product->stock * $product->cost_price,
        ];
    }
    
    /**
     * Get inventory chart data for graphing.
     */
    private function getInventoryChartData(Product $product)
    {
        $days = 90; // Last 90 days for comprehensive view
        $endDate = now();
        $startDate = now()->subDays($days);
        
        // Initialize arrays
        $labels = [];
        $dailyIn = [];
        $dailyOut = [];
        $dailyNet = [];
        $cumulativeStock = [];
        $currentStock = $product->stock;
        
        // Generate labels and data for each day
        $cumulative = $currentStock;
        for ($i = $days; $i >= 0; $i--) {
            $date = $startDate->copy()->addDays($i);
            $labels[] = $date->format('M d');
            
            // Get movements for this day
            $batchMovements = BatchStockMovement::where('product_id', $product->id)
                ->whereDate('created_at', $date->format('Y-m-d'))
                ->get();
            
            $regularMovements = StockMovement::where('product_id', $product->id)
                ->whereDate('created_at', $date->format('Y-m-d'))
                ->get();
            
            $dayIn = $batchMovements->where('movement_type', 'in')->sum('quantity') +
                    $regularMovements->where('type', 'in')->sum('qty_change');
            
            $dayOut = abs($batchMovements->where('movement_type', 'out')->sum('quantity')) +
                     abs($regularMovements->where('type', 'out')->sum('qty_change'));
            
            $dayNet = $dayIn - $dayOut;
            $cumulative -= $dayNet; // Subtract net to go backwards
            
            $dailyIn[] = $dayIn;
            $dailyOut[] = $dayOut;
            $dailyNet[] = $dayNet;
            $cumulativeStock[] = $cumulative;
        }
        
        // Reverse cumulative stock to show from past to present
        $cumulativeStock = array_reverse($cumulativeStock);
        $dailyIn = array_reverse($dailyIn);
        $dailyOut = array_reverse($dailyOut);
        $dailyNet = array_reverse($dailyNet);
        
        // Get movement type distribution
        $batchTypeDist = BatchStockMovement::where('product_id', $product->id)
            ->select('movement_type', DB::raw('SUM(ABS(quantity)) as total_quantity'))
            ->groupBy('movement_type')
            ->get()
            ->pluck('total_quantity', 'movement_type');
        
        $regularTypeDist = StockMovement::where('product_id', $product->id)
            ->select('type as movement_type', DB::raw('SUM(ABS(qty_change)) as total_quantity'))
            ->groupBy('type')
            ->get()
            ->pluck('total_quantity', 'movement_type');
        
        $typeDistribution = [
            'in' => ($batchTypeDist['in'] ?? 0) + ($regularTypeDist['in'] ?? 0),
            'out' => ($batchTypeDist['out'] ?? 0) + ($regularTypeDist['out'] ?? 0),
        ];
        
        // Get reason breakdown
        $reasons = BatchStockMovement::where('product_id', $product->id)
            ->select('reason', DB::raw('SUM(ABS(quantity)) as total_quantity'))
            ->groupBy('reason')
            ->orderByDesc('total_quantity')
            ->limit(10)
            ->get();
        
        // Get hourly distribution
        $hourlyDistribution = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $movements = BatchStockMovement::where('product_id', $product->id)
                ->whereRaw('HOUR(created_at) = ?', [$hour])
                ->count();
            
            $hourlyDistribution[] = [
                'hour' => sprintf('%02d:00', $hour),
                'movements' => $movements
            ];
        }
        
        return [
            'daily' => [
                'labels' => array_reverse($labels),
                'in' => $dailyIn,
                'out' => $dailyOut,
                'net' => $dailyNet,
                'cumulative' => $cumulativeStock,
            ],
            'typeDistribution' => $typeDistribution,
            'reasonDistribution' => $reasons,
            'hourlyDistribution' => $hourlyDistribution,
            'stock_timeline' => $this->getStockTimeline($product, 30),
        ];
    }
    
    /**
     * Get inventory trends.
     */
    private function getInventoryTrends(Product $product)
    {
        // Weekly trends
        $weeklyTrends = [];
        for ($i = 11; $i >= 0; $i--) {
            $start = now()->subWeeks($i)->startOfWeek();
            $end = now()->subWeeks($i)->endOfWeek();
            
            $batchMovements = BatchStockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $regularMovements = StockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $weekIn = $batchMovements->where('movement_type', 'in')->sum('quantity') +
                     $regularMovements->where('type', 'in')->sum('qty_change');
            
            $weekOut = abs($batchMovements->where('movement_type', 'out')->sum('quantity')) +
                       abs($regularMovements->where('type', 'out')->sum('qty_change'));
            
            $weeklyTrends[] = [
                'week' => $start->format('M d') . ' - ' . $end->format('M d'),
                'in' => $weekIn,
                'out' => $weekOut,
                'net' => $weekIn - $weekOut,
            ];
        }
        
        // Monthly trends
        $monthlyTrends = [];
        for ($i = 11; $i >= 0; $i--) {
            $start = now()->subMonths($i)->startOfMonth();
            $end = now()->subMonths($i)->endOfMonth();
            
            $batchMovements = BatchStockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $regularMovements = StockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $monthIn = $batchMovements->where('movement_type', 'in')->sum('quantity') +
                       $regularMovements->where('type', 'in')->sum('qty_change');
            
            $monthOut = abs($batchMovements->where('movement_type', 'out')->sum('quantity')) +
                        abs($regularMovements->where('type', 'out')->sum('qty_change'));
            
            $monthlyTrends[] = [
                'month' => $start->format('M Y'),
                'in' => $monthIn,
                'out' => $monthOut,
                'net' => $monthIn - $monthOut,
            ];
        }
        
        // Calculate turnover rate
        $turnoverRate = $this->calculateTurnoverRate($product);
        
        return [
            'weekly' => $weeklyTrends,
            'monthly' => $monthlyTrends,
            'turnover_rate' => $turnoverRate,
        ];
    }
    
    /**
     * Calculate inventory turnover rate.
     */
    private function calculateTurnoverRate(Product $product)
    {
        $lastYear = now()->subYear();
        
        $totalOut = BatchStockMovement::where('product_id', $product->id)
            ->where('movement_type', 'out')
            ->where('created_at', '>=', $lastYear)
            ->sum(DB::raw('ABS(quantity)'));
        
        $totalOut += StockMovement::where('product_id', $product->id)
            ->where('type', 'out')
            ->where('created_at', '>=', $lastYear)
            ->sum(DB::raw('ABS(qty_change)'));
        
        $avgStock = Product::where('id', $product->id)
            ->select(DB::raw('AVG(stock) as avg_stock'))
            ->first()
            ->avg_stock ?? 1;
        
        $turnoverRate = $avgStock > 0 ? $totalOut / $avgStock : 0;
        
        return [
            'rate' => round($turnoverRate, 2),
            'times_per_year' => round($turnoverRate, 1),
            'days_inventory' => $turnoverRate > 0 ? round(365 / $turnoverRate, 1) : 0,
            'total_out' => $totalOut,
            'avg_stock' => round($avgStock, 2),
        ];
    }
    
    /**
     * Get monthly inventory summary.
     */
    private function getMonthlyInventorySummary(Product $product)
    {
        $summary = [];
        
        for ($i = 5; $i >= 0; $i--) {
            $start = now()->subMonths($i)->startOfMonth();
            $end = now()->subMonths($i)->endOfMonth();
            
            $batchMovements = BatchStockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $regularMovements = StockMovement::where('product_id', $product->id)
                ->whereBetween('created_at', [$start, $end])
                ->get();
            
            $monthIn = $batchMovements->where('movement_type', 'in')->sum('quantity') +
                       $regularMovements->where('type', 'in')->sum('qty_change');
            
            $monthOut = abs($batchMovements->where('movement_type', 'out')->sum('quantity')) +
                        abs($regularMovements->where('type', 'out')->sum('qty_change'));
            
            $summary[] = [
                'month' => $start->format('M Y'),
                'in' => $monthIn,
                'out' => $monthOut,
                'net' => $monthIn - $monthOut,
                'movement_count' => $batchMovements->count() + $regularMovements->count(),
                'avg_daily_movement' => $end->diffInDays($start) > 0 ? 
                    round(($monthIn + $monthOut) / $end->diffInDays($start), 2) : 0,
            ];
        }
        
        return $summary;
    }
    
    /**
     * Get stock timeline for sparkline chart.
     */
    private function getStockTimeline(Product $product, $days = 30)
    {
        $timeline = [];
        $currentStock = $product->stock;
        
        for ($i = $days; $i >= 0; $i--) {
            $date = now()->subDays($i);
            
            $netChange = BatchStockMovement::where('product_id', $product->id)
                ->whereDate('created_at', $date->format('Y-m-d'))
                ->get()
                ->sum(function($movement) {
                    return $movement->movement_type == 'in' ? $movement->quantity : -$movement->quantity;
                });
            
            $netChange += StockMovement::where('product_id', $product->id)
                ->whereDate('created_at', $date->format('Y-m-d'))
                ->get()
                ->sum(function($movement) {
                    return $movement->type == 'in' ? $movement->qty_change : -$movement->qty_change;
                });
            
            $timeline[] = [
                'date' => $date->format('M d'),
                'stock' => max(0, $currentStock - $netChange), // Prevent negative stock
            ];
            
            $currentStock -= $netChange;
        }
        
        return $timeline;
    }
    
    /**
     * Export inventory history.
     */
    public function exportInventoryHistory(Product $product, Request $request)
    {
        try {
            $query = $this->getCombinedInventoryMovementsQuery($product);
            
            // Apply filters
            if ($request->has('start_date')) {
                $query->where('created_at', '>=', $request->start_date);
            }
            
            if ($request->has('end_date')) {
                $query->where('created_at', '<=', $request->end_date);
            }
            
            if ($request->has('movement_type')) {
                if ($request->movement_type == 'in') {
                    $query->where(function($q) {
                        $q->where('movement_type', 'IN')
                          ->orWhere('movement_type', 'in');
                    });
                } else {
                    $query->where(function($q) {
                        $q->where('movement_type', 'OUT')
                          ->orWhere('movement_type', 'out');
                    });
                }
            }
            
            $movements = $query->orderBy('created_at', 'desc')->get();
            
            $fileName = 'inventory-history-' . $product->sku . '-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($movements, $product) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Date', 'Time', 'Movement Source', 'Type', 'Quantity', 'Stock Before',
                    'Stock After', 'Reason', 'Batch Number', 'Reference', 'Notes', 'User'
                ]);
                
                // Data
                foreach ($movements as $movement) {
                    $stockBefore = $movement->quantity_before ?? 'N/A';
                    $stockAfter = isset($movement->quantity_before) ? 
                        ($movement->quantity_before + $movement->quantity) : 'N/A';
                    
                    fputcsv($file, [
                        $movement->created_at->format('Y-m-d'),
                        $movement->created_at->format('H:i:s'),
                        ucfirst($movement->movement_source),
                        $movement->movement_type == 'in' || $movement->movement_type == 'IN' ? 'IN' : 'OUT',
                        $movement->movement_type == 'in' || $movement->movement_type == 'IN' ? 
                            $movement->quantity : -$movement->quantity,
                        $stockBefore,
                        $stockAfter,
                        $movement->reason,
                        isset($movement->batch) ? $movement->batch->batch_number : 'N/A',
                        $movement->reference_number ?? 'N/A',
                        $movement->notes ?? 'N/A',
                        $movement->user->name ?? 'System',
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Export inventory history error: ' . $e->getMessage());
            return redirect()->route('products.inventory-history', $product)
                ->with('error', 'Failed to export inventory history.');
        }
    }
    
    /**
     * Get combined inventory movements query for export.
     */
    private function getCombinedInventoryMovementsQuery(Product $product)
    {
        // Get batch movements
        $batchMovements = BatchStockMovement::where('product_id', $product->id)
            ->with(['batch', 'user'])
            ->select([
                'id',
                'batch_id',
                'product_id',
                'movement_type',
                'quantity',
                'quantity_before',
                'reason',
                'reference_number',
                'notes',
                'user_id',
                'created_at',
                DB::raw('NULL as stock_movement_id'),
                DB::raw("'batch' as movement_source")
            ]);
        
        // Get regular stock movements
        $regularMovements = StockMovement::where('product_id', $product->id)
            ->with(['user'])
            ->select([
                'id',
                DB::raw('NULL as batch_id'),
                'product_id',
                DB::raw('UPPER(type) as movement_type'),
                'qty_change as quantity',
                'qty_before as quantity_before',
                'reason',
                DB::raw('NULL as reference_number'),
                'notes',
                'user_id',
                'created_at',
                DB::raw('id as stock_movement_id'),
                DB::raw("'regular' as movement_source")
            ]);
        
        // Return combined query
        return $batchMovements->union($regularMovements);
    }

    /**
     * Get inventory dashboard summary.
     */
    public function getInventorySummary(Product $product)
    {
        try {
            $stats = $this->getInventoryStatistics($product);
            $chartData = $this->getInventoryChartData($product);
            $trends = $this->getInventoryTrends($product);
            
            return response()->json([
                'success' => true,
                'stats' => $stats,
                'daily_chart' => $chartData['daily'],
                'type_distribution' => $chartData['typeDistribution'],
                'turnover_rate' => $trends['turnover_rate'],
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get inventory summary error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load inventory summary'
            ], 500);
        }
    }

    /**
     * Get products for POS (API endpoint)
     */
    public function apiGetProducts(Request $request): JsonResponse
    {
        try {
            $categoryId = $request->input('category_id');
            $search = $request->input('search', '');
            $limit = $request->input('limit', 50);
            
            $query = Product::with(['tax', 'category'])
                ->where('status', 'active');
            
            if ($categoryId && $categoryId !== 'all') {
                $query->where('category_id', $categoryId);
            }
            
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%")
                      ->orWhere('barcode', 'like', "%{$search}%");
                });
            }
            
            $products = $query->orderBy('name')
                ->limit($limit)
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'sku' => $product->sku,
                        'barcode' => $product->barcode,
                        'name' => $product->name,
                        'sale_price' => (float) $product->sale_price,
                        'cost_price' => (float) $product->cost_price,
                        'final_price' => (float) $product->final_price,
                        'stock' => $product->stock,
                        'available_stock' => $product->available_stock,
                        'has_vat' => (bool) $product->has_vat,
                        'is_vatable' => $product->is_vatable,
                        'tax_rate' => (float) $product->tax_rate,
                        'tax_id' => $product->tax_id,
                        'tax_details' => $product->tax ? [
                            'id' => $product->tax->id,
                            'name' => $product->tax->name,
                            'rate' => (float) $product->tax->rate,
                            'code' => $product->tax->code,
                        ] : null,
                        'category_id' => $product->category_id,
                        'category_name' => $product->category ? $product->category->name : null,
                        'is_low_stock' => $product->is_low_stock,
                        'track_inventory' => $product->track_inventory,
                    ];
                });
            
            return response()->json([
                'success' => true,
                'products' => $products
            ]);
            
        } catch (\Exception $e) {
            Log::error('Product API Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load products'
            ], 500);
        }
    }

    /**
     * Get single product for POS (API endpoint)
     */
    public function apiGetProduct($id): JsonResponse
    {
        try {
            $product = Product::with(['tax', 'category'])
                ->where('id', $id)
                ->where('status', 'active')
                ->first();
            
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'error' => 'Product not found'
                ], 404);
            }
            
            $productData = [
                'id' => $product->id,
                'sku' => $product->sku,
                'barcode' => $product->barcode,
                'name' => $product->name,
                'sale_price' => (float) $product->sale_price,
                'cost_price' => (float) $product->cost_price,
                'final_price' => (float) $product->final_price,
                'stock' => $product->stock,
                'available_stock' => $product->available_stock,
                'has_vat' => (bool) $product->has_vat,
                'is_vatable' => $product->is_vatable,
                'tax_rate' => (float) $product->tax_rate,
                'tax_id' => $product->tax_id,
                'tax_details' => $product->tax ? [
                    'id' => $product->tax->id,
                    'name' => $product->tax->name,
                    'rate' => (float) $product->tax->rate,
                    'code' => $product->tax->code,
                ] : null,
                'category_id' => $product->category_id,
                'category_name' => $product->category ? $product->category->name : null,
                'is_low_stock' => $product->is_low_stock,
                'track_inventory' => $product->track_inventory,
            ];
            
            return response()->json([
                'success' => true,
                'product' => $productData
            ]);
            
        } catch (\Exception $e) {
            Log::error('Product API Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load product'
            ], 500);
        }
    }

    /**
     * Update product tax settings in bulk
     */
    public function bulkUpdateTax(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_ids' => 'required|array',
                'product_ids.*' => 'exists:products,id',
                'has_vat' => 'required|boolean',
                'tax_id' => 'nullable|required_if:has_vat,1|exists:taxes,id',
            ]);

            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $updatedCount = 0;

            foreach ($products as $product) {
                $updateData = [
                    'has_vat' => $validated['has_vat'],
                ];

                if ($validated['has_vat']) {
                    $updateData['tax_id'] = $validated['tax_id'];
                } else {
                    $updateData['tax_id'] = null;
                }

                $product->update($updateData);
                $updatedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Tax settings updated for {$updatedCount} products.",
                'updated_count' => $updatedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk update tax error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tax settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get product tax report
     */
    public function taxReport(Request $request)
    {
        try {
            $query = Product::with(['tax', 'category', 'brand'])
                ->where('status', 'active');

            if ($request->has('tax_id')) {
                $query->where('tax_id', $request->tax_id);
            }

            if ($request->has('has_vat') && $request->has_vat != '') {
                $query->where('has_vat', $request->has_vat == '1');
            }

            $products = $query->orderBy('name')->paginate(20)->withQueryString();

            // Statistics
            $stats = [
                'total_taxable' => Product::where('has_vat', true)->count(),
                'total_exempt' => Product::where('has_vat', false)->count(),
                'by_tax_rate' => Tax::withCount(['products' => function($q) {
                    $q->where('status', 'active');
                }])->where('is_active', true)->get(),
            ];

            $taxes = Tax::where('is_active', true)->get();

            return view('products.tax-report', compact('products', 'stats', 'taxes'));
            
        } catch (\Exception $e) {
            Log::error('Tax report error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to load tax report.');
        }
    }
/**
 * Search product by barcode for POS
 */
/**
 * Search product by barcode for POS
 */
public function findByBarcode(Request $request, $barcode)
{
    try {
        // Decode URL-encoded barcode
        $barcode = urldecode($barcode);
        
        $product = Product::with(['category', 'tax', 'activeBatches'])
            ->where(function($query) use ($barcode) {
                $query->where('barcode', $barcode)
                      ->orWhere('sku', $barcode)
                      ->orWhere('ref_number', $barcode);
            })
            ->where('status', 'active')
            ->first();
        
        if (!$product) {
            return response()->json([
                'success' => false,
                'error' => 'Product not found'
            ], 404);
        }
        
        // Check stock
        if ($product->track_inventory && $product->available_stock <= 0) {
            return response()->json([
                'success' => false,
                'error' => 'Out of stock',
                'product' => [
                    'id' => $product->id,
                    'name' => $product->name,
                    'stock' => $product->stock
                ]
            ]);
        }
        
        // Format response for POS
        $productData = [
            'id' => $product->id,
            'sku' => $product->sku,
            'barcode' => $product->barcode,
            'name' => $product->name,
            'sale_price' => (float) $product->sale_price,
            'cost_price' => (float) $product->cost_price,
            'final_price' => (float) $product->final_price,
            'stock' => $product->stock,
            'available_stock' => $product->available_stock,
            'has_vat' => (bool) $product->has_vat,
            'tax_rate' => (float) $product->tax_rate,
            'tax_id' => $product->tax_id,
            'has_expiry' => (bool) $product->has_expiry,
            'track_batches' => (bool) $product->track_batches,
            'batches' => $product->track_batches ? $product->activeBatches->map(function($batch) {
                return [
                    'id' => $batch->id,
                    'batch_number' => $batch->batch_number,
                    'expiry_date' => $batch->expiry_date ? $batch->expiry_date->format('Y-m-d') : null,
                    'current_quantity' => $batch->current_quantity,
                    'batch_sale_price' => (float) $batch->batch_sale_price,
                ];
            }) : [],
            'is_low_stock' => $product->is_low_stock,
            'category' => $product->category ? $product->category->name : null,
        ];
        
        return response()->json([
            'success' => true,
            'product' => $productData
        ]);
        
    } catch (\Exception $e) {
        Log::error('Barcode search error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Search failed'], 500);
    }
}

/**
 * API endpoint for POS product search with preview
 */
public function apiSearchProducts(Request $request)
{
    try {
        $search = $request->input('q', '');
        $categoryId = $request->input('category_id');
        $limit = $request->input('limit', 20);
        
        $query = Product::with(['category', 'tax'])
            ->where('status', 'active');
        
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        if ($categoryId && $categoryId !== 'all') {
            $query->where('category_id', $categoryId);
        }
        
        $products = $query->orderBy('name')
            ->limit($limit)
            ->get()
            ->map(function ($product) {
                return [
                    'id' => $product->id,
                    'sku' => $product->sku,
                    'barcode' => $product->barcode,
                    'name' => $product->name,
                    'sale_price' => (float) $product->sale_price,
                    'final_price' => (float) $product->final_price,
                    'stock' => $product->stock,
                    'available_stock' => $product->available_stock,
                    'has_vat' => (bool) $product->has_vat,
                    'tax_rate' => (float) $product->tax_rate,
                    'category_id' => $product->category_id,
                    'category_name' => $product->category ? $product->category->name : null,
                    'is_low_stock' => $product->is_low_stock,
                    'low_stock_threshold' => $product->low_stock_threshold,
                    'image_url' => $product->image_path ? Storage::url($product->image_path) : null,
                ];
            });
        
        return response()->json([
            'success' => true,
            'products' => $products,
            'count' => $products->count()
        ]);
        
    } catch (\Exception $e) {
        Log::error('Product search API error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Search failed'], 500);
    }
}

/**
 * Download product import template.
 */
public function downloadTemplate()
{
    try {
        $fileName = 'product-import-template-' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=$fileName",
        ];

        $callback = function() {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Headers
            fputcsv($file, [
                'SKU*', 'Name*', 'Barcode', 'Category', 'Brand', 'Supplier',
                'Cost Price*', 'Sale Price*', 'Stock*', 'Minimum Stock',
                'Reorder Point', 'Maximum Stock', 'Tax Rate', 'Has Expiry (Yes/No)',
                'Expiry Warning Days', 'Description', 'Short Description'
            ]);
            
            // Example data
            fputcsv($file, [
                'SKU-001', 'Sample Product', '123456789012', 'Electronics', 'BrandX', 'SupplierA',
                '50.00', '100.00', '100', '10', '20', '200', '16%', 'No', '30',
                'Sample product description', 'Short description'
            ]);
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
        
    } catch (\Exception $e) {
        Log::error('Download template error: ' . $e->getMessage());
        return redirect()->route('products.import')
            ->with('error', 'Failed to download template.');
    }
}

}