<?php

namespace App\Http\Controllers\POS;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class POSHealthController extends Controller
{
    /**
     * Get POS health status
     */
    public function health(Request $request)
    {
        $data = [
            'status' => 'healthy',
            'timestamp' => now()->toISOString(),
            'services' => [
                'database' => $this->checkDatabase(),
                'cache' => $this->checkCache(),
                'storage' => $this->checkStorage(),
            ],
            'system' => [
                'memory_usage' => memory_get_usage(true) / 1024 / 1024,
                'memory_limit' => ini_get('memory_limit'),
                'execution_time' => microtime(true) - LARAVEL_START,
            ]
        ];
        
        return response()->json($data);
    }
    
    /**
     * Detailed health report
     */
    public function report(Request $request)
    {
        $report = [
            'health_check' => $this->health($request)->getData(),
            'pos_components' => [
                'cart_system' => $this->checkCartSystem(),
                'payment_system' => $this->checkPaymentSystem(),
                'product_system' => $this->checkProductSystem(),
            ],
            'last_errors' => $this->getLastErrors(),
            'recommendations' => $this->getRecommendations(),
        ];
        
        return response()->json($report);
    }
    
    /**
     * Log POS errors
     */
    public function logError(Request $request)
    {
        $validated = $request->validate([
            'error' => 'required|string',
            'context' => 'nullable|array',
            'severity' => 'nullable|in:low,medium,high,critical'
        ]);
        
        Log::error('POS Error: ' . $validated['error'], [
            'context' => $validated['context'] ?? [],
            'severity' => $validated['severity'] ?? 'medium',
            'user_id' => auth()->id(),
            'url' => $request->fullUrl()
        ]);
        
        return response()->json(['success' => true]);
    }
    
    private function checkDatabase()
    {
        try {
            DB::connection()->getPdo();
            return [
                'status' => 'connected',
                'version' => DB::connection()->getPdo()->getAttribute(\PDO::ATTR_SERVER_VERSION)
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'disconnected',
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function checkCache()
    {
        try {
            Cache::put('health_check', 'ok', 10);
            return Cache::get('health_check') === 'ok' 
                ? ['status' => 'working'] 
                : ['status' => 'failed'];
        } catch (\Exception $e) {
            return ['status' => 'failed', 'error' => $e->getMessage()];
        }
    }
    
    private function checkStorage()
    {
        try {
            $path = storage_path('logs');
            return [
                'status' => is_writable($path) ? 'writable' : 'readonly',
                'free_space' => disk_free_space($path) / 1024 / 1024 / 1024 . ' GB'
            ];
        } catch (\Exception $e) {
            return ['status' => 'failed', 'error' => $e->getMessage()];
        }
    }
    
    private function checkCartSystem()
    {
        try {
            // Check if cart session exists
            $hasCart = session()->has('pos_cart') || Cache::has('pos_cart_' . auth()->id());
            return ['status' => $hasCart ? 'active' : 'inactive'];
        } catch (\Exception $e) {
            return ['status' => 'error', 'error' => $e->getMessage()];
        }
    }
    
    private function checkPaymentSystem()
    {
        try {
            // Check if payment methods are available
            $methods = \App\Models\PaymentMethod::where('is_active', true)->count();
            return ['status' => $methods > 0 ? 'available' : 'unavailable', 'count' => $methods];
        } catch (\Exception $e) {
            return ['status' => 'error', 'error' => $e->getMessage()];
        }
    }
    
    private function checkProductSystem()
    {
        try {
            $products = \App\Models\Product::where('status', 'active')->count();
            return ['status' => $products > 0 ? 'available' : 'unavailable', 'count' => $products];
        } catch (\Exception $e) {
            return ['status' => 'error', 'error' => $e->getMessage()];
        }
    }
    
    private function getLastErrors()
    {
        $logFile = storage_path('logs/laravel.log');
        if (!file_exists($logFile)) {
            return [];
        }
        
        $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $posErrors = [];
        
        foreach (array_slice($lines, -50) as $line) {
            if (strpos($line, 'pos') !== false || strpos($line, 'POS') !== false) {
                $posErrors[] = substr($line, 0, 200);
            }
        }
        
        return array_slice($posErrors, -5);
    }
    
    private function getRecommendations()
    {
        $recommendations = [];
        
        // Check PHP memory
        $memoryLimit = ini_get('memory_limit');
        $memoryLimitBytes = $this->convertToBytes($memoryLimit);
        if ($memoryLimitBytes < 256 * 1024 * 1024) { // Less than 256MB
            $recommendations[] = "Increase PHP memory_limit from {$memoryLimit} to at least 256M";
        }
        
        // Check max_execution_time
        $execTime = ini_get('max_execution_time');
        if ($execTime < 120) {
            $recommendations[] = "Increase max_execution_time from {$execTime} to at least 120";
        }
        
        return $recommendations;
    }
    
    private function convertToBytes($value)
    {
        $value = trim($value);
        $last = strtolower($value[strlen($value)-1]);
        $value = (int) $value;
        
        switch($last) {
            case 'g':
                $value *= 1024;
            case 'm':
                $value *= 1024;
            case 'k':
                $value *= 1024;
        }
        
        return $value;
    }
}