<?php

namespace App\Http\Controllers;

use App\Models\MpesaTransaction;
use App\Models\Sale;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class MpesaTransactionController extends Controller
{
    /**
     * Get M-Pesa transactions with filters
     */
    public function index(Request $request)
    {
        try {
            $query = MpesaTransaction::with(['sale', 'customer'])
                ->orderBy('transaction_date', 'desc');
            
            // Apply filters
            if ($search = $request->input('search')) {
                $query->where(function($q) use ($search) {
                    $q->where('transaction_id', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('reference', 'like', "%{$search}%")
                      ->orWhereHas('sale', function($q2) use ($search) {
                          $q2->where('invoice_no', 'like', "%{$search}%");
                      })
                      ->orWhereHas('customer', function($q3) use ($search) {
                          $q3->where('name', 'like', "%{$search}%")
                             ->orWhere('phone', 'like', "%{$search}%");
                      });
                });
            }
            
            if ($status = $request->input('status')) {
                $query->where('status', $status);
            }
            
            if ($date = $request->input('date')) {
                $query->whereDate('transaction_date', $date);
            }
            
            if ($request->boolean('unmatched')) {
                $query->whereNull('sale_id')->where('status', 'completed');
            }
            
            if ($phone = $request->input('phone')) {
                $normalizedPhone = $this->formatPhoneNumber($phone);
                $query->where('phone', $normalizedPhone);
            }
            
            $perPage = $request->input('per_page', 20);
            $transactions = $query->paginate($perPage);
            
            return response()->json([
                'success' => true,
                'transactions' => $transactions->items(),
                'pagination' => [
                    'current_page' => $transactions->currentPage(),
                    'last_page' => $transactions->lastPage(),
                    'per_page' => $transactions->perPage(),
                    'total' => $transactions->total(),
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get M-Pesa transactions error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get transactions'], 500);
        }
    }
    
    /**
     * Get transactions by phone number
     */
    public function getByPhone($phone)
    {
        try {
            $normalizedPhone = $this->formatPhoneNumber($phone);
            
            $transactions = MpesaTransaction::with(['sale', 'customer'])
                ->where('phone', $normalizedPhone)
                ->where('status', 'completed')
                ->orderBy('transaction_date', 'desc')
                ->limit(10)
                ->get();
            
            // Separate matched and unmatched
            $matched = $transactions->whereNotNull('sale_id');
            $unmatched = $transactions->whereNull('sale_id');
            
            return response()->json([
                'success' => true,
                'phone' => $normalizedPhone,
                'total_transactions' => $transactions->count(),
                'matched' => $matched->values(),
                'unmatched' => $unmatched->values(),
                'stats' => [
                    'total_amount' => $transactions->sum('amount'),
                    'matched_amount' => $matched->sum('amount'),
                    'unmatched_amount' => $unmatched->sum('amount'),
                    'last_transaction' => $transactions->first()->transaction_date ?? null,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get transactions by phone error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get transactions'], 500);
        }
    }
    
    /**
     * Get M-Pesa statistics
     */
    public function stats()
    {
        try {
            $today = now()->format('Y-m-d');
            $weekStart = now()->startOfWeek()->format('Y-m-d');
            $monthStart = now()->startOfMonth()->format('Y-m-d');
            
            $stats = [
                'today' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', $today)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', $today)->sum('amount'),
                    'matched' => MpesaTransaction::whereDate('transaction_date', $today)->whereNotNull('sale_id')->count(),
                    'unmatched' => MpesaTransaction::whereDate('transaction_date', $today)->whereNull('sale_id')->count(),
                ],
                'week' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', '>=', $weekStart)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', '>=', $weekStart)->sum('amount'),
                ],
                'month' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', '>=', $monthStart)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', '>=', $monthStart)->sum('amount'),
                ],
                'total' => [
                    'count' => MpesaTransaction::count(),
                    'amount' => MpesaTransaction::sum('amount'),
                    'matched_count' => MpesaTransaction::whereNotNull('sale_id')->count(),
                    'unmatched_count' => MpesaTransaction::whereNull('sale_id')->where('status', 'completed')->count(),
                ]
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get M-Pesa stats error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get stats'], 500);
        }
    }
    
    /**
     * Get dashboard statistics
     */
    public function dashboardStats()
    {
        try {
            $today = now()->format('Y-m-d');
            
            $stats = [
                'today_total' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->sum('amount'),
                'today_count' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->count(),
                'unmatched_today' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->whereNull('sale_id')
                    ->count(),
                'recent_unmatched' => MpesaTransaction::whereNull('sale_id')
                    ->where('status', 'completed')
                    ->where('transaction_date', '>=', now()->subHours(6))
                    ->orderBy('transaction_date', 'desc')
                    ->limit(5)
                    ->get()
                    ->map(function($transaction) {
                        return [
                            'id' => $transaction->id,
                            'transaction_id' => $transaction->transaction_id,
                            'phone' => $transaction->phone,
                            'amount' => $transaction->amount,
                            'transaction_date' => $transaction->transaction_date->format('H:i:s'),
                            'full_name' => $transaction->full_name,
                        ];
                    }),
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get dashboard stats error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get dashboard stats'], 500);
        }
    }
    
    /**
     * Check for pending M-Pesa payment
     */
    public function checkPendingPayment(Request $request)
    {
        try {
            $validated = $request->validate([
                'phone' => 'required|string',
                'amount' => 'required|numeric|min:0',
                'reference' => 'nullable|string',
                'within_minutes' => 'nullable|integer|min:1|max:60'
            ]);
            
            $normalizedPhone = $this->formatPhoneNumber($validated['phone']);
            $withinMinutes = $validated['within_minutes'] ?? 10;
            
            // Look for matching transactions in the last X minutes
            $since = now()->subMinutes($withinMinutes);
            
            $transactions = MpesaTransaction::where('phone', $normalizedPhone)
                ->where('status', 'completed')
                ->whereNull('sale_id')
                ->where('transaction_date', '>=', $since)
                ->orderBy('transaction_date', 'desc')
                ->get();
            
            // Find exact match
            $exactMatch = $transactions->first(function($transaction) use ($validated) {
                return abs($transaction->amount - $validated['amount']) < 0.01;
            });
            
            if ($exactMatch) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'exact',
                    'transaction' => $exactMatch,
                    'message' => 'Exact M-Pesa payment found!'
                ]);
            }
            
            // Check for any transaction (for manual matching)
            if ($transactions->count() > 0) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'multiple',
                    'transactions' => $transactions->values(),
                    'message' => 'Found ' . $transactions->count() . ' M-Pesa transactions for this phone'
                ]);
            }
            
            return response()->json([
                'success' => true,
                'found' => false,
                'message' => 'No matching M-Pesa payment found'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Check pending payment error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to check payment'], 500);
        }
    }
    
    /**
     * Match transaction with sale
     */
    public function match(Request $request, $id)
    {
        DB::beginTransaction();
        
        try {
            $validated = $request->validate([
                'sale_id' => 'required|exists:sales,id'
            ]);
            
            $transaction = MpesaTransaction::findOrFail($id);
            $sale = Sale::with('customer')->findOrFail($validated['sale_id']);
            
            if ($transaction->sale_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction already matched with sale #' . $transaction->sale_id
                ], 400);
            }
            
            // Update transaction
            $transaction->update([
                'sale_id' => $sale->id,
                'customer_id' => $sale->customer_id,
                'status' => 'matched',
                'matched_at' => now()
            ]);
            
            // Update sale payment status if not already paid
            if ($sale->payment_status !== 'paid') {
                $sale->update([
                    'payment_status' => 'paid',
                    'payment_reference' => $transaction->transaction_id,
                    'paid_at' => now()
                ]);
                
                // Create payment record
                $sale->payments()->create([
                    'amount' => $transaction->amount,
                    'payment_method' => 'mpesa',
                    'reference' => $transaction->transaction_id,
                    'transaction_id' => $transaction->transaction_id,
                    'phone' => $transaction->phone,
                    'status' => 'completed',
                    'payment_date' => $transaction->transaction_date,
                    'received_by' => auth()->id(),
                    'notes' => 'Matched with M-Pesa transaction ' . $transaction->transaction_id
                ]);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Transaction matched successfully',
                'transaction' => $transaction,
                'sale' => $sale
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Match transaction error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to match transaction'], 500);
        }
    }
    
    /**
     * Bulk match transactions
     */
    public function bulkMatch(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validated = $request->validate([
                'matches' => 'required|array',
                'matches.*.transaction_id' => 'required|exists:mpesa_transactions,id',
                'matches.*.sale_id' => 'required|exists:sales,id'
            ]);
            
            $results = [];
            $successCount = 0;
            
            foreach ($validated['matches'] as $match) {
                try {
                    $transaction = MpesaTransaction::find($match['transaction_id']);
                    $sale = Sale::find($match['sale_id']);
                    
                    if (!$transaction->sale_id) {
                        $transaction->update([
                            'sale_id' => $sale->id,
                            'customer_id' => $sale->customer_id,
                            'status' => 'matched',
                            'matched_at' => now()
                        ]);
                        
                        $successCount++;
                        $results[] = [
                            'transaction_id' => $transaction->id,
                            'sale_id' => $sale->id,
                            'success' => true,
                            'message' => 'Matched successfully'
                        ];
                    } else {
                        $results[] = [
                            'transaction_id' => $transaction->id,
                            'sale_id' => $sale->id,
                            'success' => false,
                            'message' => 'Transaction already matched'
                        ];
                    }
                } catch (\Exception $e) {
                    $results[] = [
                        'transaction_id' => $match['transaction_id'],
                        'sale_id' => $match['sale_id'],
                        'success' => false,
                        'message' => $e->getMessage()
                    ];
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'results' => $results,
                'success_count' => $successCount,
                'failed_count' => count($validated['matches']) - $successCount,
                'message' => "Successfully matched {$successCount} transactions"
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk match error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to bulk match transactions'], 500);
        }
    }
    
    /**
     * Format phone number to 2547XXXXXXXX format
     */
    private function formatPhoneNumber($phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($phone) === 9 && strpos($phone, '0') !== 0) {
            return '254' . $phone;
        }
        
        if (strlen($phone) === 10 && strpos($phone, '0') === 0) {
            return '254' . substr($phone, 1);
        }
        
        if (strlen($phone) === 12 && strpos($phone, '254') === 0) {
            return $phone;
        }
        
        return $phone;
    }

    // Add these methods to App\Http\Controllers\MpesaTransactionController.php

/**
 * Show single transaction
 */
public function show($id)
{
    try {
        $transaction = MpesaTransaction::with(['sale', 'customer'])->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'transaction' => $transaction
        ]);
        
    } catch (\Exception $e) {
        Log::error('Get transaction error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Transaction not found'], 404);
    }
}

/**
 * Reverse a transaction
 */
public function reverse($id)
{
    DB::beginTransaction();
    
    try {
        $transaction = MpesaTransaction::findOrFail($id);
        
        // Check if can be reversed
        if ($transaction->status !== 'completed') {
            return response()->json([
                'success' => false,
                'error' => 'Only completed transactions can be reversed'
            ], 400);
        }
        
        // Mark as reversed
        $transaction->update([
            'status' => 'reversed',
            'reversed_at' => now(),
            'reversed_by' => auth()->id()
        ]);
        
        // If linked to sale, update sale payment status
        if ($transaction->sale_id) {
            $sale = Sale::find($transaction->sale_id);
            if ($sale) {
                $sale->update([
                    'payment_status' => 'pending',
                    'payment_reference' => null,
                    'paid_at' => null
                ]);
            }
        }
        
        DB::commit();
        
        return response()->json([
            'success' => true,
            'message' => 'Transaction reversed successfully',
            'transaction' => $transaction
        ]);
        
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Reverse transaction error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Failed to reverse transaction'], 500);
    }
}

/**
 * Daily report
 */
public function dailyReport(Request $request)
{
    try {
        $date = $request->input('date', now()->format('Y-m-d'));
        
        $transactions = MpesaTransaction::whereDate('transaction_date', $date)
            ->where('status', 'completed')
            ->orderBy('transaction_date', 'desc')
            ->get();
        
        $summary = [
            'total_amount' => $transactions->sum('amount'),
            'total_count' => $transactions->count(),
            'matched_count' => $transactions->whereNotNull('sale_id')->count(),
            'unmatched_count' => $transactions->whereNull('sale_id')->count(),
            'matched_amount' => $transactions->whereNotNull('sale_id')->sum('amount'),
            'unmatched_amount' => $transactions->whereNull('sale_id')->sum('amount'),
        ];
        
        return response()->json([
            'success' => true,
            'date' => $date,
            'summary' => $summary,
            'transactions' => $transactions
        ]);
        
    } catch (\Exception $e) {
        Log::error('Daily report error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Failed to generate report'], 500);
    }
}

/**
 * Unmatched transactions report
 */
public function unmatchedReport(Request $request)
{
    try {
        $days = $request->input('days', 7);
        $since = now()->subDays($days);
        
        $transactions = MpesaTransaction::whereNull('sale_id')
            ->where('status', 'completed')
            ->where('transaction_date', '>=', $since)
            ->orderBy('transaction_date', 'desc')
            ->get();
        
        return response()->json([
            'success' => true,
            'days' => $days,
            'count' => $transactions->count(),
            'total_amount' => $transactions->sum('amount'),
            'transactions' => $transactions
        ]);
        
    } catch (\Exception $e) {
        Log::error('Unmatched report error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Failed to generate report'], 500);
    }
}

/**
 * Reconciliation report
 */
public function reconciliation(Request $request)
{
    try {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        $transactions = MpesaTransaction::whereBetween('transaction_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->get();
        
        $reconciliation = [
            'period' => [
                'start' => $startDate,
                'end' => $endDate
            ],
            'total_transactions' => $transactions->count(),
            'total_amount' => $transactions->sum('amount'),
            'matched' => [
                'count' => $transactions->whereNotNull('sale_id')->count(),
                'amount' => $transactions->whereNotNull('sale_id')->sum('amount'),
            ],
            'unmatched' => [
                'count' => $transactions->whereNull('sale_id')->count(),
                'amount' => $transactions->whereNull('sale_id')->sum('amount'),
            ],
            'by_day' => $transactions->groupBy(function($item) {
                return $item->transaction_date->format('Y-m-d');
            })->map(function($dayTransactions) {
                return [
                    'count' => $dayTransactions->count(),
                    'amount' => $dayTransactions->sum('amount'),
                    'matched' => $dayTransactions->whereNotNull('sale_id')->count(),
                    'unmatched' => $dayTransactions->whereNull('sale_id')->count(),
                ];
            })
        ];
        
        return response()->json([
            'success' => true,
            'reconciliation' => $reconciliation
        ]);
        
    } catch (\Exception $e) {
        Log::error('Reconciliation error: ' . $e->getMessage());
        return response()->json(['success' => false, 'error' => 'Failed to generate reconciliation'], 500);
    }
}
}