<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\Sale;
use App\Models\Product;
use App\Models\Customer;
use App\Models\Purchase;
use App\Models\ActivityLog;
use App\Models\User;

class DashboardController extends Controller
{
    /**
     * Display the dashboard.
     */
    public function index()
    {
        return view('dashboard.index');
    }

    /**
     * Get dashboard statistics data.
     */
    public function getDashboardData(Request $request)
    {
        $period = $request->get('period', 30); // Default to 30 days
        
        $startDate = Carbon::now()->subDays($period);
        $previousStartDate = Carbon::now()->subDays($period * 2);
        $previousEndDate = Carbon::now()->subDays($period);
        
        // Total Revenue
        $totalRevenue = Sale::where('created_at', '>=', $startDate)->sum('grand_total');
        $previousRevenue = Sale::whereBetween('created_at', [$previousStartDate, $previousEndDate])->sum('grand_total');
        $revenueChange = $previousRevenue > 0 ? round((($totalRevenue - $previousRevenue) / $previousRevenue) * 100, 1) : 0;
        
        // Total Sales Count
        $totalSales = Sale::where('created_at', '>=', $startDate)->count();
        $previousSales = Sale::whereBetween('created_at', [$previousStartDate, $previousEndDate])->count();
        $salesChange = $previousSales > 0 ? round((($totalSales - $previousSales) / $previousSales) * 100, 1) : 0;
        
        // Total Customers
        $totalCustomers = Customer::count();
        $previousCustomers = Customer::where('created_at', '<', $startDate)->count();
        $customersChange = $previousCustomers > 0 ? round((($totalCustomers - $previousCustomers) / $previousCustomers) * 100, 1) : 0;
        
        // Total Products
        $totalProducts = Product::count();
        $previousProducts = Product::where('created_at', '<', $startDate)->count();
        $productsChange = $previousProducts > 0 ? round((($totalProducts - $previousProducts) / $previousProducts) * 100, 1) : 0;
        
        // Today's Sales
        $todaySales = Sale::whereDate('created_at', Carbon::today())->sum('grand_total');
        $yesterdaySales = Sale::whereDate('created_at', Carbon::yesterday())->sum('grand_total');
        $todaySalesChange = $yesterdaySales > 0 ? round((($todaySales - $yesterdaySales) / $yesterdaySales) * 100, 1) : 0;
        
        // Pending Orders (adjust based on your status field)
        $pendingOrders = Sale::where('status', 'pending')->count();
        $previousPendingOrders = Sale::where('status', 'pending')->where('created_at', '<', $startDate)->count();
        $pendingOrdersChange = $previousPendingOrders > 0 ? round((($pendingOrders - $previousPendingOrders) / $previousPendingOrders) * 100, 1) : 0;
        
        // Sales overview data for chart
        $salesData = [];
        for ($i = $period - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $salesData[] = [
                'date' => $date->format('M d'),
                'sales' => Sale::whereDate('created_at', $date)->sum('grand_total')
            ];
        }
        
        // Format amounts for display with KES
        $formattedRevenue = 'KES ' . number_format($totalRevenue, 2);
        $formattedTodaySales = 'KES ' . number_format($todaySales, 2);
        
        return response()->json([
            'total_revenue' => $totalRevenue,
            'formatted_revenue' => $formattedRevenue,
            'revenue_change' => $revenueChange,
            'total_sales' => $totalSales,
            'sales_change' => $salesChange,
            'total_customers' => $totalCustomers,
            'customers_change' => $customersChange,
            'total_products' => $totalProducts,
            'products_change' => $productsChange,
            'today_sales' => $todaySales,
            'formatted_today_sales' => $formattedTodaySales,
            'today_sales_change' => $todaySalesChange,
            'pending_orders' => $pendingOrders,
            'pending_orders_change' => $pendingOrdersChange,
            'sales_overview' => [
                'labels' => array_column($salesData, 'date'),
                'series' => array_column($salesData, 'sales')
            ],
            'currency' => [
                'symbol' => 'KES ',
                'decimal_places' => 2,
                'thousands_separator' => ',',
                'decimal_separator' => '.'
            ]
        ]);
    }

    /**
     * Get recent activities.
     */
    public function getRecentActivities()
    {
        $activities = ActivityLog::with('causer')
            ->latest()
            ->take(10)
            ->get();
            
        if ($activities->isEmpty()) {
            return '<div class="text-center py-4 text-muted">No activities found</div>';
        }
        
        $html = '';
        foreach ($activities as $activity) {
            $html .= '
            <div class="activity-item d-flex align-items-center border-bottom py-2">
                <div class="activity-icon bg-light-primary text-primary rounded-circle p-2 me-3">
                    <i class="bx bx-' . $this->getActivityIcon($activity->log_name) . '"></i>
                </div>
                <div class="flex-grow-1">
                    <h6 class="mb-0">' . $activity->description . '</h6>
                    <small class="text-muted">' . $activity->created_at->diffForHumans() . '</small>
                </div>
            </div>';
        }
        
        return $html;
    }

    /**
     * Get low stock products.
     */

        public function getLowStockProducts()
        {
            $lowStockProducts = Product::where('stock', '<=', DB::raw('reorder_point'))
                ->where('status', 'active')
                ->whereNull('deleted_at')
                ->orderBy('stock', 'asc')
                ->limit(5)
                ->get();
            
            return $lowStockProducts;
        }

    /**
     * Get recent sales.
     */
    public function getRecentSales()
    {
        $recentSales = Sale::with('customer')
            ->latest()
            ->take(8)
            ->get();
            
        if ($recentSales->isEmpty()) {
            return '<tr><td colspan="5" class="text-center py-4 text-muted">No recent sales</td></tr>';
        }
        
        $html = '';
        foreach ($recentSales as $sale) {
            $statusClass = $this->getStatusClass($sale->status);
            
            $html .= '
            <tr>
                <td>
                    <a href="' . route('sales.show', $sale->id) . '" class="text-primary">
                        #' . str_pad($sale->id, 6, '0', STR_PAD_LEFT) . '
                    </a>
                </td>
                <td>' . ($sale->customer ? $sale->customer->name : 'Walk-in Customer') . '</td>
                <td>' . $sale->created_at->format('M d, Y h:i A') . '</td>
                <td class="fw-bold">KES ' . number_format($sale->grand_total, 2) . '</td>
                <td>
                    <span class="badge bg-' . $statusClass . '">' . ucfirst($sale->status) . '</span>
                </td>
            </tr>';
        }
        
        return $html;
    }

    /**
     * Get top products.
     */
    public function getTopProducts()
    {
        // Assuming you have a sales_items table
        $topProducts = DB::table('sale_items')
            ->select('product_id', DB::raw('SUM(quantity) as total_sold'))
            ->join('products', 'sale_items.product_id', '=', 'products.id')
            ->groupBy('product_id')
            ->orderByDesc('total_sold')
            ->take(5)
            ->get();
            
        if ($topProducts->isEmpty()) {
            return '<div class="text-center py-4 text-muted">No sales data available</div>';
        }
        
        $html = '';
        $counter = 1;
        foreach ($topProducts as $item) {
            $product = Product::find($item->product_id);
            if (!$product) continue;
            
            $html .= '
            <div class="row border mx-0 mb-3 py-2 radius-10 cursor-pointer">
                <div class="col-sm-6">
                    <div class="d-flex align-items-center">
                        <div class="product-img">
                            <img src="' . $product->image_url . '" alt="' . $product->name . '" 
                                 style="width: 40px; height: 40px; object-fit: cover;">
                        </div>
                        <div class="ms-2">
                            <h6 class="mb-1">' . $product->name . '</h6>
                            <p class="mb-0 text-muted">#' . $counter . ' Best Seller</p>
                        </div>
                    </div>
                </div>
                <div class="col-sm">
                    <h6 class="mb-1">' . $item->total_sold . ' Sold</h6>
                    <p class="mb-0">Stock: ' . $product->stock_quantity . '</p>
                </div>
            </div>';
            
            $counter++;
        }
        
        return $html;
    }

    /**
     * Helper function to get activity icon.
     */
    private function getActivityIcon($logName)
    {
        $icons = [
            'default' => 'bell',
            'sale' => 'cart',
            'purchase' => 'package',
            'product' => 'box',
            'customer' => 'user',
            'user' => 'user-circle'
        ];
        
        return $icons[$logName] ?? 'bell';
    }

    /**
     * Helper function to get status class.
     */
    private function getStatusClass($status)
    {
        $classes = [
            'pending' => 'warning',
            'completed' => 'success',
            'cancelled' => 'danger',
            'refunded' => 'info'
        ];
        
        return $classes[$status] ?? 'secondary';
    }
}