<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Intervention\Image\Facades\Image;

class BrandController extends Controller
{
    /**
     * Display a listing of the resource.
     */
/**
 * Display a listing of the resource.
 */
public function index(Request $request)
{
    try {
        $query = Brand::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('slug', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('contact_email', 'like', "%{$search}%")
                  ->orWhere('contact_phone', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by country
        if ($request->has('country') && $request->country != '') {
            $query->where('country', $request->country);
        }

        // Filter by featured
        if ($request->has('featured') && $request->featured == '1') {
            $query->where('is_featured', true);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');
        
        $allowedSort = ['name', 'order', 'created_at', 'status'];
        if (in_array($sortBy, $allowedSort)) {
            $query->orderBy($sortBy, $sortOrder);
        } else {
            $query->orderBy('order')->orderBy('name');
        }

        $brands = $query->paginate(20)->withQueryString();

        // Statistics
        $stats = [
            'total' => Brand::count(),
            'active' => Brand::where('status', 'active')->count(),
            'featured' => Brand::where('is_featured', true)->count(),
            'total_products' => DB::table('products')
                ->join('brands', 'products.brand_id', '=', 'brands.id')
                ->whereNotNull('products.brand_id')
                ->count(),
        ];

        // Get unique countries from existing brands
        $countries = Brand::select('country')
            ->whereNotNull('country')
            ->where('country', '!=', '')
            ->distinct()
            ->orderBy('country')
            ->pluck('country');

        return view('brands.index', compact('brands', 'stats', 'countries'));
        
    } catch (\Exception $e) {
        Log::error('Brand index error: ' . $e->getMessage());
        return redirect()->route('dashboard')
            ->with('error', 'Failed to load brands. Please try again.');
    }
}
    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::where('status', 'active')->get();
        $countries = $this->getCountries();
        
        return view('brands.create', compact('categories', 'countries'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:brands,name',
                'slug' => 'nullable|string|max:255|unique:brands,slug',
                'description' => 'nullable|string',
                'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'website' => 'nullable|url|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:20',
                'status' => 'required|in:active,inactive',
                'country' => 'nullable|string|max:100',
                'established_year' => 'nullable|integer|min:1800|max:' . date('Y'),
                'meta_title' => 'nullable|string|max:255',
                'meta_description' => 'nullable|string|max:500',
                'meta_keywords' => 'nullable|string|max:500',
                'order' => 'nullable|integer|min:0',
                'is_featured' => 'boolean',
                'category_ids' => 'nullable|array',
                'category_ids.*' => 'exists:categories,id',
            ], [
                'name.unique' => 'This brand name already exists.',
                'slug.unique' => 'This slug already exists.',
                'logo.image' => 'The logo must be an image file.',
                'logo.max' => 'The logo must not exceed 2MB.',
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $data = $validator->validated();

            // Handle logo upload
            if ($request->hasFile('logo')) {
                $logoPath = $this->uploadLogo($request->file('logo'));
                $data['logo'] = $logoPath;
            }

            // Set is_featured
            $data['is_featured'] = $request->has('is_featured');

            // Create brand
            $brand = Brand::create($data);

            // Sync categories
            if ($request->has('category_ids')) {
                $brand->categories()->sync($request->input('category_ids'));
            }

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($brand)
                ->log('created brand');

            DB::commit();

            return redirect()->route('brands.index')
                ->with('success', 'Brand created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            throw $e;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Brand store error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to create brand: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Brand $brand)
    {
        try {
            $brand->load(['categories', 'products' => function($query) {
                $query->with(['category', 'unit'])
                    ->orderBy('created_at', 'desc')
                    ->limit(10);
            }]);

            $productsCount = $brand->products()->count();
            $activeProductsCount = $brand->products()->where('status', 'active')->count();
            $totalStockValue = $brand->products()->sum(DB::raw('cost_price * stock'));

            $statistics = [
                'total_products' => $productsCount,
                'active_products' => $activeProductsCount,
                'inactive_products' => $productsCount - $activeProductsCount,
                'total_stock_value' => $totalStockValue,
                'avg_product_price' => $brand->products()->avg('sale_price'),
                'low_stock_products' => $brand->products()
                    ->whereColumn('stock', '<=', 'minimum_stock')
                    ->where('stock', '>', 0)
                    ->count(),
                'out_of_stock_products' => $brand->products()
                    ->where('stock', '<=', 0)
                    ->count(),
            ];

            return view('brands.show', compact('brand', 'statistics'));
            
        } catch (\Exception $e) {
            Log::error('Brand show error: ' . $e->getMessage());
            return redirect()->route('brands.index')
                ->with('error', 'Failed to load brand details.');
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Brand $brand)
    {
        $categories = Category::where('status', 'active')->get();
        $selectedCategories = $brand->categories->pluck('id')->toArray();
        $countries = $this->getCountries();
        
        return view('brands.edit', compact('brand', 'categories', 'selectedCategories', 'countries'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Brand $brand)
    {
        try {
            DB::beginTransaction();

            $validator = Validator::make($request->all(), [
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('brands')->ignore($brand->id),
                ],
                'slug' => [
                    'nullable',
                    'string',
                    'max:255',
                    Rule::unique('brands')->ignore($brand->id),
                ],
                'description' => 'nullable|string',
                'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'website' => 'nullable|url|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:20',
                'status' => 'required|in:active,inactive',
                'country' => 'nullable|string|max:100',
                'established_year' => 'nullable|integer|min:1800|max:' . date('Y'),
                'meta_title' => 'nullable|string|max:255',
                'meta_description' => 'nullable|string|max:500',
                'meta_keywords' => 'nullable|string|max:500',
                'order' => 'nullable|integer|min:0',
                'is_featured' => 'boolean',
                'category_ids' => 'nullable|array',
                'category_ids.*' => 'exists:categories,id',
                'remove_logo' => 'nullable|boolean',
            ], [
                'name.unique' => 'This brand name already exists.',
                'slug.unique' => 'This slug already exists.',
                'logo.image' => 'The logo must be an image file.',
                'logo.max' => 'The logo must not exceed 2MB.',
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $oldData = $brand->toArray();
            $data = $validator->validated();

            // Handle logo removal
            if ($request->has('remove_logo') && $request->remove_logo == '1') {
                if ($brand->logo) {
                    Storage::delete('public/brands/' . $brand->logo);
                }
                $data['logo'] = null;
            }

            // Handle logo upload
            if ($request->hasFile('logo')) {
                // Delete old logo if exists
                if ($brand->logo) {
                    Storage::delete('public/brands/' . $brand->logo);
                }
                
                $logoPath = $this->uploadLogo($request->file('logo'));
                $data['logo'] = $logoPath;
            }

            // Set is_featured
            $data['is_featured'] = $request->has('is_featured');

            // Update brand
            $brand->update($data);

            // Sync categories
            if ($request->has('category_ids')) {
                $brand->categories()->sync($request->input('category_ids'));
            } else {
                $brand->categories()->detach();
            }

            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($brand)
                ->withProperties([
                    'old' => $oldData,
                    'new' => $data
                ])
                ->log('updated brand');

            DB::commit();

            return redirect()->route('brands.index')
                ->with('success', 'Brand updated successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            throw $e;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Brand update error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to update brand: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Brand $brand)
    {
        try {
            DB::beginTransaction();

            // Check if brand has products
            if ($brand->products()->count() > 0) {
                return redirect()->route('brands.index')
                    ->with('error', 'Cannot delete brand because it has products. Please update or delete those products first.');
            }

            // Delete logo if exists
            if ($brand->logo) {
                Storage::delete('public/brands/' . $brand->logo);
            }

            // Detach categories
            $brand->categories()->detach();

            // Log activity before deletion
            activity()
                ->causedBy(auth()->user())
                ->performedOn($brand)
                ->log('deleted brand');

            $brand->delete();

            DB::commit();

            return redirect()->route('brands.index')
                ->with('success', 'Brand deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Brand destroy error: ' . $e->getMessage());
            return redirect()->route('brands.index')
                ->with('error', 'Failed to delete brand. Please try again.');
        }
    }

    /**
     * Search brands for AJAX requests.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q');
            $limit = $request->input('limit', 10);
            $checkSlug = $request->input('check_slug', false);

            if ($checkSlug) {
                // Check if slug exists
                $exists = Brand::where('slug', $search)->exists();
                return response()->json(['exists' => $exists]);
            }

            $brands = Brand::where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('slug', 'like', "%{$search}%");
                })
                ->where('status', 'active')
                ->select('id', 'name', 'slug', 'logo')
                ->limit($limit)
                ->get();

            return response()->json($brands);
            
        } catch (\Exception $e) {
            Log::error('Brand search error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /**
     * Bulk delete brands.
     */
    public function bulkDelete(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'brand_ids' => 'required|array',
                'brand_ids.*' => 'exists:brands,id',
            ]);

            $brands = Brand::whereIn('id', $validated['brand_ids'])->get();
            $deletedCount = 0;
            $failedCount = 0;
            $errors = [];

            foreach ($brands as $brand) {
                // Check if brand has products
                if ($brand->products()->count() > 0) {
                    $failedCount++;
                    $errors[] = "Brand '{$brand->name}' has products.";
                    continue;
                }

                // Delete logo if exists
                if ($brand->logo) {
                    Storage::delete('public/brands/' . $brand->logo);
                }

                // Detach categories
                $brand->categories()->detach();

                // Log activity before deletion
                activity()
                    ->causedBy(auth()->user())
                    ->performedOn($brand)
                    ->log('bulk deleted brand');

                $brand->delete();
                $deletedCount++;
            }

            DB::commit();

            $message = "Deleted {$deletedCount} brands successfully.";
            if ($failedCount > 0) {
                $message .= " {$failedCount} brands could not be deleted.";
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'deleted_count' => $deletedCount,
                'failed_count' => $failedCount,
                'errors' => $errors,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk delete brands error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete brands: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk update brands.
     */
    public function bulkUpdate(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'brand_ids' => 'required|array',
                'brand_ids.*' => 'exists:brands,id',
                'data' => 'required|array',
                'data.status' => 'nullable|in:active,inactive',
                'data.is_featured' => 'nullable|boolean',
            ]);

            $brands = Brand::whereIn('id', $validated['brand_ids'])->get();
            $updatedCount = 0;

            foreach ($brands as $brand) {
                $brand->update(array_filter($validated['data']));
                $updatedCount++;
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Updated {$updatedCount} brands successfully.",
                'updated_count' => $updatedCount,
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk update brands error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update brands: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export brands.
     */
    public function export(Request $request)
    {
        try {
            $brands = Brand::with(['categories'])
                ->orderBy('name')
                ->get();

            $fileName = 'brands-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];

            $callback = function() use ($brands) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'ID', 'Name', 'Slug', 'Description', 'Logo', 'Website',
                    'Contact Email', 'Contact Phone', 'Status', 'Country',
                    'Established Year', 'Featured', 'Order', 'Categories', 'Created At'
                ]);
                
                // Data
                foreach ($brands as $brand) {
                    fputcsv($file, [
                        $brand->id,
                        $brand->name,
                        $brand->slug ?? '',
                        $brand->description ?? '',
                        $brand->logo ?? '',
                        $brand->website ?? '',
                        $brand->contact_email ?? '',
                        $brand->contact_phone ?? '',
                        $brand->status,
                        $brand->country ?? '',
                        $brand->established_year ?? '',
                        $brand->is_featured ? 'Yes' : 'No',
                        $brand->order,
                        $brand->categories->pluck('name')->implode(', '),
                        $brand->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Brand export error: ' . $e->getMessage());
            return redirect()->route('brands.index')
                ->with('error', 'Failed to export brands.');
        }
    }

    /**
     * Get brands for AJAX select.
     */
    public function getBrands(Request $request)
    {
        try {
            $search = $request->input('search', '');
            
            $brands = Brand::where('status', 'active')
                ->where(function($query) use ($search) {
                    if (!empty($search)) {
                        $query->where('name', 'like', "%{$search}%")
                              ->orWhere('description', 'like', "%{$search}%");
                    }
                })
                ->orderBy('name')
                ->limit(20)
                ->get(['id', 'name', 'logo']);
            
            $formattedBrands = $brands->map(function($brand) {
                return [
                    'id' => $brand->id,
                    'text' => $brand->name,
                    'logo' => $brand->logo_url,
                ];
            });
            
            return response()->json($formattedBrands);
            
        } catch (\Exception $e) {
            Log::error('Get brands error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /**
     * Brand statistics.
     */
    public function statistics()
    {
        try {
            // Brand statistics
            $stats = [
                'total_brands' => Brand::count(),
                'active_brands' => Brand::where('status', 'active')->count(),
                'inactive_brands' => Brand::where('status', 'inactive')->count(),
                'featured_brands' => Brand::where('is_featured', true)->count(),
                'brands_with_logo' => Brand::whereNotNull('logo')->count(),
                'brands_without_logo' => Brand::whereNull('logo')->count(),
                'avg_products_per_brand' => DB::table('products')
                    ->select(DB::raw('AVG(product_count) as avg_products'))
                    ->fromSub(function($query) {
                        $query->select('brand_id', DB::raw('COUNT(*) as product_count'))
                            ->from('products')
                            ->whereNotNull('brand_id')
                            ->groupBy('brand_id');
                    }, 'brand_products')
                    ->first()->avg_products ?? 0,
            ];

            // Country distribution
            $countryDistribution = Brand::select('country', DB::raw('COUNT(*) as count'))
                ->whereNotNull('country')
                ->groupBy('country')
                ->orderByDesc('count')
                ->limit(10)
                ->get();

            // Featured brands
            $featuredBrands = Brand::where('is_featured', true)
                ->withCount('products')
                ->orderBy('order')
                ->limit(10)
                ->get();

            // Brands with most products
            $topBrandsByProducts = Brand::withCount(['products' => function($query) {
                $query->where('status', 'active');
            }])
                ->orderByDesc('products_count')
                ->limit(10)
                ->get();

            return view('brands.statistics', compact(
                'stats', 
                'countryDistribution', 
                'featuredBrands', 
                'topBrandsByProducts'
            ));
            
        } catch (\Exception $e) {
            Log::error('Brand statistics error: ' . $e->getMessage());
            return redirect()->route('brands.index')
                ->with('error', 'Failed to load brand statistics.');
        }
    }

    /**
     * Upload and process brand logo.
     */
    private function uploadLogo($file)
    {
        $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        
        // Create directory if it doesn't exist
        $directory = 'public/brands';
        if (!Storage::exists($directory)) {
            Storage::makeDirectory($directory);
        }
        
        // Process and save image
        $image = Image::make($file);
        
        // Resize if needed
        if ($image->width() > 800 || $image->height() > 800) {
            $image->resize(800, 800, function ($constraint) {
                $constraint->aspectRatio();
                $constraint->upsize();
            });
        }
        
        // Save to storage
        $image->save(storage_path('app/' . $directory . '/' . $fileName));
        
        return $fileName;
    }

    /**
     * Get list of countries.
     */
    private function getCountries()
    {
        return [
            'US' => 'United States',
            'CA' => 'Canada',
            'GB' => 'United Kingdom',
            'AU' => 'Australia',
            'DE' => 'Germany',
            'FR' => 'France',
            'IT' => 'Italy',
            'ES' => 'Spain',
            'JP' => 'Japan',
            'CN' => 'China',
            'IN' => 'India',
            'BR' => 'Brazil',
            'MX' => 'Mexico',
            'RU' => 'Russia',
            'ZA' => 'South Africa',
            'NG' => 'Nigeria',
            'KE' => 'Kenya',
            'EG' => 'Egypt',
            'SA' => 'Saudi Arabia',
            'AE' => 'United Arab Emirates',
            'TR' => 'Turkey',
            'KR' => 'South Korea',
            'SG' => 'Singapore',
            'MY' => 'Malaysia',
            'TH' => 'Thailand',
            'VN' => 'Vietnam',
            'ID' => 'Indonesia',
            'PH' => 'Philippines',
            'PK' => 'Pakistan',
            'BD' => 'Bangladesh',
            'LK' => 'Sri Lanka',
            'NP' => 'Nepal',
            'BT' => 'Bhutan',
            'MV' => 'Maldives',
        ];
    }
}