<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ETIMSCredential;
use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ETIMSController extends Controller
{
    public function getConfig()
    {
        return response()->json([
            'success' => true,
            'config' => config('etims')
        ]);
    }

    public function getActiveCredentials()
    {
        try {
            $credentials = ETIMSCredential::active()->first();
            
            if (!$credentials) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active eTIMS credentials found',
                    'credentials' => null
                ]);
            }
            
            return response()->json([
                'success' => true,
                'credentials' => [
                    'id' => $credentials->id,
                    'name' => $credentials->name,
                    'tin' => $credentials->tin_masked,
                    'environment' => $credentials->environment,
                    'api_url' => $credentials->api_url,
                    'branch_id' => $credentials->branch_id,
                    'branch_name' => $credentials->branch_name
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('ETIMS credentials error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load eTIMS credentials'
            ], 500);
        }
    }

    public function submitReceipt(Request $request)
    {
        try {
            $data = $request->validate([
                'invoice_no' => 'required|string',
                'customer' => 'required|array',
                'items' => 'required|array',
                'totals' => 'required|array',
                'payment' => 'required|array'
            ]);
            
            // Get active credentials
            $credentials = ETIMSCredential::active()->first();
            
            if (!$credentials) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active eTIMS credentials configured'
                ], 400);
            }
            
            // Prepare eTIMS data
            $etimsData = $this->prepareETIMSData($data, $credentials);
            
            // Submit to eTIMS
            $response = $this->submitToETIMS($etimsData, $credentials);
            
            if ($response['success']) {
                // Update sale record with eTIMS info
                Sale::where('invoice_no', $data['invoice_no'])
                    ->update([
                        'etims_submitted' => true,
                        'etims_submission_id' => $response['submission_id'],
                        'etims_qr_code' => $response['qr_code'] ?? null,
                        'etims_submitted_at' => now(),
                        'etims_response' => json_encode($response)
                    ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Successfully submitted to eTIMS',
                    'submission_id' => $response['submission_id'],
                    'qr_code' => $response['qr_code'] ?? null,
                    'verification_url' => $response['verification_url'] ?? null
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'eTIMS submission failed',
                    'error' => $response['error']
                ], 400);
            }
            
        } catch (\Exception $e) {
            Log::error('ETIMS submission error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'eTIMS submission failed',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    private function prepareETIMSData($data, $credentials)
    {
        // Implement eTIMS data preparation logic
        // This is a simplified example
        return [
            'tin' => $credentials->getDecryptedTin(),
            'bhfId' => $credentials->branch_id,
            'dvcSrlNo' => 'DVC' . time(),
            'invoiceNo' => $data['invoice_no'],
            'invoiceType' => '01', // 01: Invoice
            'invoiceDate' => date('Ymd'),
            'invoiceTime' => date('His'),
            'customerName' => $data['customer']['name'],
            'customerTin' => $data['customer']['pin'] ?? '',
            'customerMobileNum' => $data['customer']['phone'] ?? '',
            'taxType' => '01', // 01: VAT
            'supplyTypeCode' => '01', // 01: Goods
            'paymentType' => $this->getETIMSPaymentType($data['payment']['method']),
            'items' => $this->prepareETIMSItems($data['items'])
        ];
    }

    private function getETIMSPaymentType($method)
    {
        $types = [
            'cash' => '01',
            'mpesa' => '02',
            'card' => '03',
            'credit' => '04',
            'bank' => '05'
        ];
        
        return $types[$method] ?? '01';
    }

    private function prepareETIMSItems($items)
    {
        return array_map(function($item) {
            return [
                'itemCode' => $item['sku'] ?? 'ITEM' . $item['id'],
                'itemName' => substr($item['name'], 0, 100),
                'qty' => $item['quantity'],
                'unitPrice' => $item['unit_price'],
                'taxRate' => $item['tax_rate'] ?? 16,
                'discount' => 0,
                'itemTotal' => $item['total']
            ];
        }, $items);
    }

    private function submitToETIMS($data, $credentials)
    {
        // This is a mock implementation
        // In production, you would make actual API calls to KRA eTIMS
        
        Log::info('Mock eTIMS submission:', $data);
        
        // Simulate API call delay
        sleep(1);
        
        return [
            'success' => true,
            'submission_id' => 'ETIMS-' . time() . '-' . rand(1000, 9999),
            'qr_code' => 'data:image/png;base64,mock_qr_code',
            'verification_url' => 'https://etims.kra.go.ke/verify/' . 'ETIMS-' . time() . '-' . rand(1000, 9999),
            'message' => 'Successfully submitted to eTIMS sandbox'
        ];
    }
}