<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Api\ApiBaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardApiController extends ApiBaseController
{
    /**
     * Get today's statistics
     */
    public function getTodayStats()
    {
        try {
            $today = Carbon::today();
            $yesterday = Carbon::yesterday();
            
            // Today's sales
            $todaySales = DB::table('sales')
                ->whereDate('created_at', $today)
                ->where('status', 'completed')
                ->count();
            
            $todayRevenue = DB::table('sales')
                ->whereDate('created_at', $today)
                ->where('status', 'completed')
                ->sum('total_amount');
            
            // Yesterday's sales for comparison
            $yesterdayRevenue = DB::table('sales')
                ->whereDate('created_at', $yesterday)
                ->where('status', 'completed')
                ->sum('total_amount');
            
            $revenueChange = $yesterdayRevenue > 0 
                ? (($todayRevenue - $yesterdayRevenue) / $yesterdayRevenue) * 100 
                : 100;
            
            // Today's customers
            $todayCustomers = DB::table('sales')
                ->whereDate('created_at', $today)
                ->where('status', 'completed')
                ->distinct('customer_id')
                ->count('customer_id');
            
            // Average sale
            $avgSale = $todaySales > 0 
                ? $todayRevenue / $todaySales 
                : 0;
            
            // Low stock products
            $lowStockProducts = DB::table('products')
                ->where('stock', '<=', DB::raw('low_stock_threshold'))
                ->where('stock', '>', 0)
                ->count();
            
            // Out of stock products
            $outOfStockProducts = DB::table('products')
                ->where('stock', '<=', 0)
                ->count();
            
            // Top selling products today
            $topProducts = DB::table('sale_items')
                ->join('products', 'sale_items.product_id', '=', 'products.id')
                ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
                ->whereDate('sales.created_at', $today)
                ->where('sales.status', 'completed')
                ->select(
                    'products.id',
                    'products.name',
                    'products.sku',
                    DB::raw('SUM(sale_items.quantity) as total_quantity'),
                    DB::raw('SUM(sale_items.total_price) as total_revenue')
                )
                ->groupBy('products.id', 'products.name', 'products.sku')
                ->orderBy('total_quantity', 'desc')
                ->limit(5)
                ->get();
            
            // Recent sales
            $recentSales = DB::table('sales')
                ->leftJoin('customers', 'sales.customer_id', '=', 'customers.id')
                ->whereDate('sales.created_at', $today)
                ->where('sales.status', 'completed')
                ->select(
                    'sales.id',
                    'sales.invoice_number',
                    'sales.total_amount',
                    'sales.payment_method',
                    'sales.created_at',
                    'customers.name as customer_name'
                )
                ->orderBy('sales.created_at', 'desc')
                ->limit(10)
                ->get();
            
            return $this->successResponse([
                'sales_count' => $todaySales,
                'revenue' => round($todayRevenue, 2),
                'revenue_change' => round($revenueChange, 2),
                'average_sale' => round($avgSale, 2),
                'customers_count' => $todayCustomers,
                'low_stock_products' => $lowStockProducts,
                'out_of_stock_products' => $outOfStockProducts,
                'top_products' => $topProducts,
                'recent_sales' => $recentSales,
                'date' => $today->format('Y-m-d')
            ], 'Today\'s statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to load today\'s statistics');
        }
    }

    /**
     * Get dashboard statistics
     */
    public function getDashboardStats()
    {
        try {
            $today = Carbon::today();
            $startOfMonth = Carbon::now()->startOfMonth();
            $startOfWeek = Carbon::now()->startOfWeek();
            
            // Monthly statistics
            $monthlySales = DB::table('sales')
                ->where('created_at', '>=', $startOfMonth)
                ->where('status', 'completed')
                ->count();
            
            $monthlyRevenue = DB::table('sales')
                ->where('created_at', '>=', $startOfMonth)
                ->where('status', 'completed')
                ->sum('total_amount');
            
            // Weekly statistics
            $weeklySales = DB::table('sales')
                ->where('created_at', '>=', $startOfWeek)
                ->where('status', 'completed')
                ->count();
            
            $weeklyRevenue = DB::table('sales')
                ->where('created_at', '>=', $startOfWeek)
                ->where('status', 'completed')
                ->sum('total_amount');
            
            // Total products
            $totalProducts = DB::table('products')->count();
            
            // Total customers
            $totalCustomers = DB::table('customers')->count();
            
            // Total suppliers
            $totalSuppliers = DB::table('suppliers')->count();
            
            // Pending purchases
            $pendingPurchases = DB::table('purchases')
                ->where('status', 'pending')
                ->count();
            
            // Sales by payment method
            $paymentMethods = DB::table('sales')
                ->where('created_at', '>=', $startOfMonth)
                ->where('status', 'completed')
                ->select(
                    'payment_method',
                    DB::raw('COUNT(*) as count'),
                    DB::raw('SUM(total_amount) as amount')
                )
                ->groupBy('payment_method')
                ->get();
            
            // Daily revenue for chart (last 7 days)
            $dailyRevenue = [];
            for ($i = 6; $i >= 0; $i--) {
                $date = Carbon::today()->subDays($i);
                $revenue = DB::table('sales')
                    ->whereDate('created_at', $date)
                    ->where('status', 'completed')
                    ->sum('total_amount');
                
                $dailyRevenue[] = [
                    'date' => $date->format('Y-m-d'),
                    'day' => $date->format('D'),
                    'revenue' => round($revenue, 2)
                ];
            }
            
            return $this->successResponse([
                'monthly' => [
                    'sales' => $monthlySales,
                    'revenue' => round($monthlyRevenue, 2)
                ],
                'weekly' => [
                    'sales' => $weeklySales,
                    'revenue' => round($weeklyRevenue, 2)
                ],
                'totals' => [
                    'products' => $totalProducts,
                    'customers' => $totalCustomers,
                    'suppliers' => $totalSuppliers
                ],
                'pending_purchases' => $pendingPurchases,
                'payment_methods' => $paymentMethods,
                'daily_revenue' => $dailyRevenue,
                'period' => [
                    'month' => $startOfMonth->format('F Y'),
                    'week' => 'Week ' . $startOfWeek->weekOfYear
                ]
            ], 'Dashboard statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to load dashboard statistics');
        }
    }



}