<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\UnifiedNumberSequence;
use Illuminate\Support\Facades\Log;

class CleanupNumberSequences extends Command
{
    protected $signature = 'numbers:cleanup 
                            {--days=30 : Clean sequences older than X days}
                            {--dry-run : Show what would be deleted without actually deleting}';
    
    protected $description = 'Clean up old number sequences';
    
    public function handle()
    {
        $days = $this->option('days');
        $dryRun = $this->option('dry-run');
        $date = now()->subDays($days)->format('Y-m-d');
        
        $this->info("Cleaning up number sequences older than {$days} days...");
        $this->info("Cutoff date: {$date}");
        
        $query = UnifiedNumberSequence::where('date', '<', $date)
            ->where('type', 'invoice_receipt');
        
        $count = $query->count();
        
        if ($count === 0) {
            $this->info("No old number sequences found to clean up.");
            return 0;
        }
        
        if ($dryRun) {
            $this->info("[DRY RUN] Would delete {$count} old number sequences:");
            
            $sequences = $query->limit(10)->get();
            foreach ($sequences as $sequence) {
                $this->line("  - {$sequence->date}: {$sequence->prefix} (Last: {$sequence->last_number})");
            }
            
            if ($count > 10) {
                $this->line("  ... and " . ($count - 10) . " more");
            }
            
            return 0;
        }
        
        $deleted = $query->delete();
        
        $this->info("✅ Successfully deleted {$deleted} old number sequences.");
        Log::info("Cleaned up {$deleted} old number sequences (older than {$days} days).");
        
        return 0;
    }
}